# 
# Copyright (c) 2009-2025 The TEBA Project. All rights reserved.
# 
# Redistribution and use in source, with or without modification, are
# permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# Author: Atsuhi Yoshida

use warnings;
use strict;

#use GraphViz;
use JSON;

use TEBA2JSON;
use SymbolTable;
use TEBA2TEXT;

#########################################################################
=pod

=encoding utf8

=head1 AST::NODE とは

TEBA の JSON 形式の AST を、
属性参照のためのメソッドを持つノードの構成として扱うライブラリ

=cut

package AST::NODE;

use Data::Dumper;  # for debug

sub new {
    return bless {};
}

=pod

=head1 AST の構築に関するメソッド

=head2 build_from_tokens

字句列から AST を構成する。字句列をそのまま AST にするのではなく、
TEBA2JSON で最適化した AST を構成する。これは cparse.pl -j で
構成する AST と同じである。

=head3 使用例

  my $token_array = [ <> ];
  my $ast = AST::NODE->build_from_tokens($token_array);

=cut
sub build_from_tokens {
    my ($class, $tokens) = @_;
    # 字句列の AST を木構造に変換

    my $ast = TEBA2JSON->new($tokens)->json->tree;

    # 記号表の作成
    $ast->{sym} = SymbolTable->new->with_standard_global_table
        ->analyze($ast)->root_tree;

    return $class->bless_ast($ast);
}

=pod

=head2 build_from_json_text

JSON形式の AST (つまり、cparse.pl -j の出力) からノードで構成される AST を構成する。
この場合、JSON形式のデータはテキストであること。

JSON に decode 済みのデータの場合には、このメソッドではなく bless_ast を使用する。

=head3 使用例

  my $json_text = join('', <>);
  my $ast = AST::NODE->build_from_json_text($json_text);
=cut
sub build_from_json_text {
    my ($class, $json_text) = @_;
    my $root = JSON->new->decode($json_text);
    return $class->bless_ast($root); # AST の各ノードを CFG::ASTNODE クラスのオブジェクトに変換
}

=pod

=head1 ノードの子要素、属性の取得のメソッド

=head2 children_all

ノードの子をすべてリストで返す。

=head3 使用例

  my @children = $node->children_all;

=cut
# すべての子供
sub children_all {
    my $self = shift;
    return grep(&isObj($_), @{$self->{e}});
}

=pod

=head2 children

引数で指定された属性の子をリストで返す。

=head3 使用例

  my @cond_children = $node->children('cond');

=cut
# 指定した属性の子供のリスト
sub children {
    my $self = shift;
    my $attr = shift;

    return () unless exists $self->{$attr};
    my $c = $self->{$attr};
    unless (ref($c) eq "ARRAY") {
        $c = [ $c ];
    }
    return map($self->{e}->[$_], @$c);
}

=pod

=head2 child

引数で指定された属性の子を返す。
その属性の子要素が1つの場合にのみ、利用できる。
複数の子要素がいる場合は実行時エラーになる。

=head3 使用例

  my $then = $if_node->child('then');

=cut
# 指定した種類の子供 (リストになっているものは扱えない)
sub child {
    my $self = shift;
    my $attr = shift;

    return undef unless exists $self->{$attr};
    my $c = $self->{$attr};
    if (ref($c) eq "ARRAY") {
        die "Can't get $attr as a child";
    }
    return $self->{e}->[$c];
}

=pod

=head2 first_child

子要素のうち最初の要素を返す。
子要素が1つしかいないときに、その子要素を取り出すのに利用できる。

=head3 使用例

  my $inner = $if_node->first_child;

=cut
sub first_child {
    my $self = shift;

    foreach (@{$self->{e}}) {
        return $_ if &isObj($_);
    }
    return undef;
}


=pod

=head2 attr

引数で指定された属性の値を返す。

=head3 使用例

  my $var_name = $var_node->attr('name');

=cut
# 指定した属性の値の取得
sub attr {
    my $self = shift;
    my $attr = shift;

    return undef unless exists $self->{$attr};
    return $self->{$attr};
}

=pod

=head2 set

引数で指定された属性の値を設定する。

=head3 使用例

  my $var_name = $var_node->set('peval', num(5));

=cut
# 指定した属性の値の設定
sub set {
    my $self = shift;
    my ($attr, $value) = @_;

    $self->{$attr} = $value;
}

=pod

=head2 append_element

引数に指定された要素を、構成要素リスト(すなわち、属性 e のリスト)に追加する。
なお、構成要素リストを変更すると、属性の値と整合しなくなる可能性があるので注意。

=head3 使用例

  $node->append_element("_HOGE <>", "_PIYO <>");

=cut

# 構成要素の最後に要素リストを加える
sub append_element {
    my $self = shift;
    push(@{$self->{e}}, @_);
}

=pod

=head2 set_element

引数に指定された要素を、構成要素リスト(すなわち、属性 e のリスト)にする。
なお、構成要素リストを変更すると、属性の値と整合しなくなる可能性があるので注意。

=head3 使用例

  $node->set_element("B_HOGE #H01 <>", "HOGE <hoge>", "E_HOGE #H01 <>");

=cut
# 構成要素を置換える
sub set_element {
    my $self = shift;
    $self->{e} = [ @_ ];
}


=pod

=head2 特定の属性の取り出し用メソッドや便利メソッド

以下に示すメソッドは、ある種の子要素や属性値を取り出すためのメソッドである。
これらをを使わなくても、child や attr を使っても取り出せるが、
このようにメソッドを定義すると、プログラムが簡潔になる。

=over

=item id : 属性 id の値を返す。

=item op_symbol : 演算子の記号を返す (属性 "sym")

=item st_cond : 制御文の条件式ノードを返す (属性 "cond" で参照されるノード)

=item uniq_varname : 変数名とスコープ情報と組み合わせたユニーク名を生成

=item is_empty : 空の要素であるかの判定 (属性 e のリストが空)

=item is_node : 引数の要素が AST::NODE かどうかを判定

=back

=cut

# 典型的な属性の取り出し専用のメソッドの例

sub id {
    return $_[0]->{id};
}

sub elem {
    return @{$_[0]->{e}};
}

sub op_symbol {
    my $self = shift;
    return $self->attr("sym");
}

sub st_cond {
    my $self = shift;
    return $self->child("cond");
}

sub uniq_varname {
    my $self = shift;
#    die "Not variable: $self->{t}." unless $self->{t} eq "ID_VF";
    die "Not variable: $self->{t}." unless $self->{t} =~ "ID_(VF|MC)";
    return $self->{name}.':'.$self->{scope};
}

sub is_empty {
    my $self = shift;
    return int(@{$self->{e}}) == 0;
}

sub is_node {
    my $self = shift;
    my $elem = shift;
    return ref($elem) eq "AST::NODE";
}

sub left_ctx {
    my $self = shift;
    if (exists $self->{left}) {
        return $self->{left};
    } else {
        return "";
    }
}

sub isObj() {
    return ref($_[0]) ne "";
}



=pod

=head1 AST の変換

=head2 str

AST から字句列を取り出し、ソースコードのテキストを構成する。

=head3 使用例

  $node->set_element("B_HOGE #H01 <>", "HOGE <hoge>", "E_HOGE #H01 <>");

=cut

sub str {
    my $self = shift;
    return &join_tokens($self->all_tokens);
}

sub all_tokens {
    my $self = shift;
    return map(&isObj($_) ? $_->all_tokens: $_, @{$self->{e}});
}

sub join_tokens {
    my @tk = @_;
    grep(s/^\w+(?:\s+#?\w+)?\s+<(.*)>$/$1/, @tk);
    return join('', map(&TEBA2TEXT::ev($_), @tk));
}

=pod

=head2 bless_ast

JSON 形式のデータを AST::NODE に変換する。

=head3 使用例

    my $root = JSON->new->decode($json_text);
    $root = $class->bless_ast($root);

=cut


# JSON 形式のAST をこのクラスのオブジェクトに変換する処理
sub bless_ast {
    my $class = shift;
    my $root = shift;
    &_bless_ast($root);
    return $root;
}

sub _bless_ast {
    my $root = shift;
    bless $root;
    map(&_bless_ast($_), $root->children_all);
}

=pod

=head2 tokens

AST を字句列にする。変換時に AST は unbless され、JSONデータに戻るので注意すること。

=head3 使用例

    print $root->tokens;

=cut

sub tokens {
    my $self = shift;
    $self->unbless_ast();
    my $tj = TEBA2JSON->new->set_json_tree($self->unbless_ast);
    return $tj->teba;
}

=pod

=head2 tokens

AST を JSON データに戻す。

=head3 使用例

  $root->unbless_ast;

=cut


# AST を元のハッシュだけの構成にしたものを返す処理
# JSON 化するときは、これが必要。
sub unbless_ast {
    my $root = shift;
    if (&isObj($root)){
        my $ret = { %$root };
        $ret->{e} = [ map(&unbless_ast($_), @{$root->{e}}) ];
        return $ret;
    } else {
        return $root; # i.e. a string
    }
}


=pod

=head2 generate_node_map


AST のノードマップ(ノードIDからノードへのハッシュ)を作る。

=cut

# node map (from id to node)の生成
sub generate_node_map {
    my $self = shift;
    return $self->{node_map} = &_generate_node_map($self);
}

sub _generate_node_map {
    my $el = shift;        # 部分木の根
    my $map = shift || {}; # マップ (引数が空のときは空のハッシュを用意)

    # 部分木の根を map に に登録
    $map->{$el->id} = $el;
    foreach ($el->children_all) {  # 子要素についても再帰的に登録
        $map = &_generate_node_map($_, $map);
    }
    return $map; # 更新した map を返す
}

=pod

=head2 node

ノードマップを使って、ノードidに対応するノードを求めて返す。

=cut

sub node {
    my $self = shift;
    my $id = shift;
    return exists $self->{node_map}->{$id} ? $self->{node_map}->{$id} : undef;
}

=pod

=head2 node_all

ノードマップを使って、すべてのノードを求めて返す。

=cut

sub node_all {
    my $self = shift;
    my $id = shift;
    return values %{$self->{node_map}};
}

=pod

=head2 remove_node_map

ノードマップを AST から削除する。JSON 化するときには削除しておく必要がある。

=cut
# node map の削除
# node map を JSON 化できないので、作ったら削除する必要がある。
# JSON化用のメソッドを定義して、自動化するべき
sub remove_node_map {
    my $self = shift;
    delete $self->{node_map};
    return $self;
}

=pod

=head2 select_node_from($query)

指定したノードを根とする部分木から、条件 $query に合う属性を持つノードのリストを返す

=cut

sub select_node_from {
    my $self = shift;
    my $query = shift;

    my @res;

    if ($query->($self)) {
        push(@res, $self);
    }

    foreach ($self->children_all) {
        push(@res, $_->select_node_from($query));
    }

    return @res;
}

1;
