# 
# Copyright (c) 2009-2025 The TEBA Project. All rights reserved.
# 
# Redistribution and use in source, with or without modification, are
# permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# Author: Atsuhi Yoshida

use warnings;
use strict;

use GraphViz;
use JSON;

use AST;

#use TEBA2JSON;

#########################################################################
# CFG class の定義

# CFG の上を動く CFG::VISITOR については、このあとの定義のコメントを参照すること。
# 解析処理は、基本的に CFG::VISITOR を使うと便利(なはず)。

package CFG;

use Data::Dumper;  # for debug

# CFG オプションの生成
sub new {
    my $class = shift;
    my $root = shift;    # AST (JSON から戻したメモリ上の木)の根
    my $opts = shift;
    my $self = bless {}; # ハッシュとしてCFGオブジェクトを生成

    $self->{opts} = $opts || { };
    $self->{opts}->{abst_level} = "E"
	unless exists $self->{opts}->{abst_level}; # デフォルトは式レベル

    # 根が指定されていたら、CFG の処理に必要なデータを生成
    $self = $self->build($root) if $root;
    return $self;
}
# オプションについて
# opts{abst_level}: { S => Statement level CFG,
#                     L => logical expression level CFG,
#                     E => expression level CFG (default) }
# opts{remove_logical_op} : removing logical operators(&&, ||)
# opts{as_shrinked_graph} : to_dot generates shrinked CFG.
# opts{literal_cond} : apply evalution of literal condition in if and while.
# opts{with_expr_node} : to create expression begin and end nodes.

# Expression level の CFG の生成
sub new_expr {
    my $class = shift;
    my $root = shift;
    return $class->new($root, {e => 1});
}

sub load {
    my $class = shift;
    my $root = shift;    # AST (JSON から戻したメモリ上の木)の根
    my $self = bless {}; # ハッシュとしてCFGオブジェクトを生成

    if (ref($root) ne "AST::NODE") {
        $root = AST::NODE->bless_ast($root);  # AST tree
    }
    $self->{ast} = $root;  # AST tree

    # AST のノード id からノードへのマップを作る
    $self->{ast}->generate_node_map;

    # CFG をメモリ上に構成し、CFGのノード id からノードへのマップを作る
    $self->{node_map} = &generate_cfg_node_map($self->{ast});

    $self->prepare_begin_end_node_name;

    return $self;
}


# CFG の処理に必要なデータの生成
sub build {
    my $self = shift;
    $self->{ast} = shift;  # AST tree

    AST::NODE->bless_ast($self->{ast}); # AST の各ノードを AST::NODE クラスのオブジェクトに変換

    # build CFG from AST
    $self->ast_traverse($self->{ast});
    $self->connectJumpNode();
    $self->remove_logical_op() if $self->{opts}->{remove_logical_op};
    $self->optimize_cfg();
    $self->add_to_ast();

    # AST のノード id からノードへのマップを作る
    $self->{ast}->generate_node_map;

    # CFG をメモリ上に構成し、CFGのノード id からノードへのマップを作る
    $self->{node_map} = &generate_cfg_node_map($self->{ast});

    $self->prepare_begin_end_node_name;

    # 補足: AST や CFG を JSON 形式で表現するために、元の AST や CFG のデータでは
    # ノードの id を使ってノード間の関係を表している。しかし、関係をたどっていくときは
    # id ではなく、ノードそのものを参照したいので、これらのマップで必要に応じて変換
    # できるようにする。次に定義する node_begin や node_end は、その例である。

    return $self;
}


# CFG の開始ノードリストを返す
sub node_begin {
    my $self = shift;
    my $name = shift;

    if ($name) {
        if (exists $self->{begin_node}->{$name}) {
            return $self->{begin_node}->{$name};
        }
        $name .= "-begin";
        if (exists $self->{begin_node}->{$name}) {
            return $self->{begin_node}->{$name};
        }
        return undef;
    }
    return values %{$self->{begin_node}};
}


# CFG の終了ノードリストを返す
sub node_end {
    my $self = shift;
    my $name = shift;

    if ($name) {
        if (exists $self->{end_node}->{$name}) {
            return $self->{end_node}->{$name};
        }
        $name .= "-end";
        if (exists $self->{end_node}->{$name}) {
            return $self->{end_node}->{$name};
        }
        return undef;
    }
    return values %{$self->{end_node}};
}

sub prepare_begin_end_node_name {
    my $self = shift;
    $self->{begin_node} = {};
    my @node_id = @{$self->{ast}->{cfg}->{begin}};
    my @node = map($self->{node_map}->{$_}, @node_id);
    foreach (@node) {
        $self->{begin_node}->{$_->label} = $_;
    }

    $self->{end_node} = {};
    @node_id = @{$self->{ast}->{cfg}->{end}};
    @node = map($self->{node_map}->{$_}, @node_id);
    foreach (@node) {
        $self->{end_node}->{$_->label} = $_;
    }
}




# すべてのノードを返す
sub node_all {
    my $self = shift;
    return values %{$self->{node_map}};
}

sub node {
    my $self = shift;
    my $id = shift;
    my $map = $self->{node_map};
    return undef unless exists $map->{$id};
    return $map->{$id};
}

# CFG の AST を返す
sub ast {
    my $self = shift;
    return $self->{ast};
}

# CFG を動く visitor オブジェクトを返す
sub visitor {
    my $self = shift;
    return CFG::VISITOR->new($self);
}

sub to_json {
    my $self = shift;
    my $ast = $self->{ast}->remove_node_map->unbless_ast;
    $self->flush_cfg_node_map_to_json($self->{node_map});
    return JSON->new->max_depth()->canonical->pretty->convert_blessed->encode($ast);
}

#--------------------------------------------------------------------#

# AST から CFG の変換

sub ast_traverse
{
    my $self = shift;
    my $el = shift;
    my $jump_tbl = shift; # a linked list of a map to node from jump type
    # 'out' => the end of function for return
    # 'brk' => the end of loop for break
    # 'cnt' => the end of loop for continue
    # 'sw' => the top of switch
    # 'p' => previous map
    # 'blk' => the end of the blocks that the visitor passes through.
    # 'Esc' => the end of the path that the visitor passes through.
    my ($begin, $end);

    if ($el->{t} eq "FUNC") {
        my $fname = $el->child('name')->str;
        $begin = $self->createNode("#func-$fname-begin", $el);
        $end = $self->createNode("#func-$fname-end", $el);
        &setNodePair($begin, $end);

        my ($arg_b, $arg_e) = $self->ast_traverse_args($el->children('arg'));
        if ($arg_b) {
            &connectNode($begin, $arg_b);
        } else {
            $arg_e = $begin;
        }

        my $jt = { 'out' => $end , 'p' => $jump_tbl };
        my ($b, $e) = $self->ast_traverse($el->child('body'), $jt);
        &connectNode($arg_e, $b);
        &connectNode($e, $end);
    } elsif ($el->{t} eq "ST_COMP") {
        $begin = $self->createNode("#comp-begin", $el);
        $end = $self->createNode("#comp-end", $el);
        &setNodePair($begin, $end);
        my $jt = { 'blk' => $end, 'p' => $jump_tbl };

        my $cur = $begin;
        foreach my $st ($el->children_all) {
            (my $b, my $e, $jt) = $self->ast_traverse($st, $jt);
            &connectNode($cur, $b);
            $cur = $e;
        }
        &connectNode($cur, $self->jump_to($jt, 'blk'));
    } elsif ($el->{t} eq "ST_IF") {
        $begin = $self->createNode("#if-begin", $el);
        $end = $self->createNode("#if-end", $el);
        &setNodePair($begin, $end);
        my $jt = { 'blk' => $end, 'p' => $jump_tbl };

        my $c = $el->child("cond");
        my @cond = $self->createNodeWithExpr("cond", $c->str, $c);
        &connectNode($begin, $cond[0]);

	my $bool = $self->eval_literal_cond($c);

	if ($bool ne "F") {
	    $jt = { 'esc' => $self->createNode("#then-", $el), 'p' => $jt };
	    my ($b, $e) = $self->ast_traverse($el->child("then"), $jt);
	    &connectNode($cond[1], $self->create_bool_node("#true", $c, $bool),
		     $self->createNode("#then-in", $el), $b);
	    &connectNode($e, $self->createNode("#then-out", $el), $end);
	}

	if ($bool ne "T") {
	    my $else_in = $self->createNode("#else-in", $el);
	    my $else_out = $self->createNode("#else-out", $el);

	    &connectNode($cond[1], $self->create_bool_node("#false", $c, $bool),
		     $else_in);

	    if (exists($el->{else})) {
		$jt = { 'esc' => $self->createNode("#else-", $el), 'p' => $jt };
		my ($b, $e) = $self->ast_traverse($el->child("else"), $jt);
		&connectNode($else_in, $b);
		&connectNode($e, $else_out);
	    } else {
		&connectNode($else_in, $else_out);
	    }
	    &connectNode($else_out, $end);
	}
    } elsif ($el->{t} eq "ST_WHILE") {
        $begin = $self->createNode("#while-begin", $el);
        $end = $self->createNode("#while-end", $el);
        &setNodePair($begin, $end);
        my $c = $el->child("cond");
        my @cond = $self->createNodeWithExpr("cond", $c->str, $c);
        &connectNode($begin, $cond[0]);

	my $bool = $self->eval_literal_cond($c);

	if ($bool ne "T") {
	    &connectNode($cond[1], $self->create_bool_node("#false", $c, $bool),
			 $self->createNode("#loop_end-in", $el),
			 $self->createNode("#loop_end-out", $el), $end);

	}
	if ($bool ne "F") {
	    my $jt = { 'brk' => $end, 'cnt' => $cond[0], 'p' => $jump_tbl };
	    $jt = { 'esc' => $self->createNode("#loop-", $el), 'p' => $jt };

	    my ($b, $e) = $self->ast_traverse($el->child("body"), $jt);
	    &connectNode($cond[1], $self->create_bool_node("#true", $c, $bool),
			 $self->createNode("#loop-in", $el), $b);
	    &connectNode($e, $self->createNode("#loop-back", $el), $cond[0]);
	}
    } elsif ($el->{t} eq "ST_FOR") {
        $begin = $self->createNode("#for-begin", $el);
        $end = $self->createNode("#for-end", $el);
        &setNodePair($begin, $end);
        my @c = $el->children("cond");

        my @init = $self->createNodeWithExpr("for_init", $c[0]->str, $c[0]);
        my @cond = $self->createNodeWithExpr("cond", $c[1]->str, $c[1]);
        my @succ = $self->createNodeWithExpr("for_succ", $c[2]->str, $c[2]);


	my $bool = @{$c[1]->{e}} ? $self->eval_literal_cond($c[1]) : "T";

        &connectNode($begin, $init[0]);
        &connectNode($init[1], $cond[0]);
	if ($bool ne "T") {
	    &connectNode($cond[1],
			 $self->create_bool_node("#false", $c[1], $bool),
			 $self->createNode("#loop_end-in", $el),
			 $self->createNode("#loop_end-out", $el), $end);
	}

	if ($bool ne "F") {
	    my $jt = { 'brk' => $end, 'cnt' => $cond[0], 'p' => $jump_tbl };
	    $jt = { 'esc' => $self->createNode("#loop-", $el), 'p' => $jt };

	    my ($b, $e) = $self->ast_traverse($el->child("body"), $jt);

	    &connectNode($cond[1], $self->create_bool_node("#true", $c[1], $bool),
			 $self->createNode("#loop-in", $el), $b);
	    &connectNode($e, $succ[0]);
	    &connectNode($succ[1], $self->createNode("#loop-back", $el), $cond[0]);
	}
    } elsif ($el->{t} eq "ST_DO") {
        $begin = $self->createNode("#do-begin", $el);
        $end = $self->createNode("#do-end", $el);
        &setNodePair($begin, $end);

        my $loop_in = $self->createNode("#loop-in", $el);
        my $c = $el->child("cond");
        my @cond = $self->createNodeWithExpr("cond", $c->str, $c);

        my $jt = { 'brk' => $end, 'cnt' => $cond[0], 'p' => $jump_tbl };
        $jt = { 'esc' => $self->createNode("#loop-", $el), 'p' => $jt };

        my ($b, $e) = $self->ast_traverse($el->child("body"), $jt);
        &connectNode($begin, $loop_in, $b);
        &connectNode($e, $self->createNode("#loop-back", $el), $cond[0]);

	my $bool = $self->eval_literal_cond($c);

	if ($bool ne "T") {
	    &connectNode($cond[1], $self->create_bool_node("#false", $c, $bool),
			 $self->createNode("#loop_end-in", $el),
			 $self->createNode("#loop_end-out", $el), $end);
	}
	if ($bool ne "F") {
	    &connectNode($cond[1], $self->create_bool_node("#true", $c, $bool),
			 $loop_in);
	}
    } elsif ($el->{t} eq "ST_SWITCH") {
        $begin = $self->createNode("#switch-begin", $el);
        $end = $self->createNode("#switch-end", $el);
        &setNodePair($begin, $end);
        my $c = $el->child("cond");
        my @cond = $self->createNodeWithExpr("cond", $c->str, $c);
        &connectNode($begin, $cond[0]);

        my $jt = { 'sw' => $cond[1], 'brk' => $end, 'p' => $jump_tbl };
        my ($b, $e) = $self->ast_traverse($el->child("body"), $jt);
        &connectNode($e, $end);

        my $has_default = 0;
        foreach my $n ($cond[1]->next) {
            if ($n->label eq "#default") {
                $has_default = 1;
                last;
            }
        }
        &connectNode($cond[1], $end) unless $has_default;
    } elsif ($el->{t} eq "ST_EXPR") {
        if (my $expr = $el->child("expr")) {
            ($begin, $end) = $self->createNodeWithExpr("expr_st", $expr->str, $expr);
            $begin->{ast_node} = $el;
            $end->{ast_node} = $el;
        } else {
            $begin = $self->createNode("#expr_st-begin", $el);
            $end = $self->createNode("#expr_st-end", $el);
            &setNodePair($begin, $end);
            &connectNode($begin, $self->createNode("#empty", $el), $end);
        }
    } elsif ($el->{t} eq "ST_JUMP") {
        if ($el->{sort} eq "return") {
            $begin = $self->createNode("#return-begin", $el);
            $end = $self->createNode("#return-end", $el);
            &setNodePair($begin, $end);
            if (my $e = $el->child("value")) {
                my @val_node = $self->createNodeWithExpr("return_val", $e->str, $e);
                &connectNode($begin, $val_node[0]);
                &connectNode($val_node[1], $end);
            } else {
                my $node = $self->createNode("#return", $el);
                &connectNode($begin, $node, $end);
            }
            &connectNode($end, $self->jump_to($jump_tbl, 'out'));
            $end = undef;
        } elsif ($el->{sort} eq "break") {
            $begin = $self->createNode("#break", $el);
            &connectNode($begin, $self->jump_to($jump_tbl, 'brk'));
        } elsif ($el->{sort} eq "continue") {
            $begin = $self->createNode("#continue", $el);
            &connectNode($begin, $self->jump_to($jump_tbl, 'cnt'));
        } elsif ($el->{sort} eq "goto") {
            my $lb = $el->{label_name} || "";
            $begin = $self->createNode("#goto $lb", $el);
            # goto のときのブロックの出入りに対応していない
            $self->saveJumpNode($begin, $jump_tbl);
        } else {
            die "Not supported yet for $el->{sort}\n";
        }
        $jump_tbl = &cleanup_switch_label($jump_tbl);
        # switch から脱出するときに、case, default ラベルのブロックを脱出するので、
        # jump_tbl からラベルを除去
    } elsif ($el->{t} eq "ST_LABELED") {
        if ($el->{sort} eq "jump") {
            my $lb = $el->{label_name} || "";
            $begin = $self->createLabelNode("label: $lb", $el, $jump_tbl);
        } elsif ($el->{sort} eq "case") {
            my $case_label = $el->child("label");
            my $case = $case_label->str;
            $begin = $self->createNode("#case_$case-in", $el);
            &connectNode($self->switch_to($jump_tbl, "#case_$case", $el), $begin);
            $jump_tbl = { 'esc' => $self->createNode("#case_$case-", $case_label),
                              'p' => $jump_tbl };
        } else  { # default
            $begin = $self->createNode("#default-in", $el);
            &connectNode($self->switch_to($jump_tbl, "#default", $el), $begin);
            $jump_tbl = { 'esc' => $self->createNode("#default-", $el),
                              'p' => $jump_tbl };
        }
        (my $b, my $e, $jump_tbl) = $self->ast_traverse($el->child("body"), $jump_tbl);
        &connectNode($begin, $b);
        $end = $e;
    } elsif($el->{t} eq "DE") {
        ($begin, $end) = $self->ast_traverse_decl($el);
    } elsif ($el->{t} eq "UNIT") {
        my @children = $el->children_all;

        $begin = $self->createNode("#unit-begin", $el);
        $end = $self->createNode("#unit-end", $el);
        &setNodePair($begin, $end);
        my $last_end = $begin;
        my $jt = { 'out' => $end, 'brk' => $end, 'cnt' => $begin,
                       'p' => $jump_tbl };
        foreach (@children) {
            my ($b, $e) = $self->ast_traverse($_, $jt);
            next unless $b;
            if ($b->{label} =~ /^#func-/) {
                push(@{$self->{begin_node}}, $b);
                push(@{$self->{end_node}}, $e);
            } else {
                &connectNode($last_end, $b);
                $last_end = $e;
            }
        }
        &connectNode($last_end, $end);
        push(@{$self->{begin_node}}, $begin);
        push(@{$self->{end_node}}, $end);

        # if (grep($_->{t} =~ /^ST_/, @children)) {
        #     $begin = $self->createNode("#unit-begin", $el);
        #     $end = $self->createNode("#unit-end", $el);
        #     &setNodePair($begin, $end);
        #     my $last_end = $begin;
        #     my $jt = { 'out' => $end, 'brk' => $end, 'cnt' => $begin,
        #                    'p' => $jump_tbl };
        #     foreach (@children) {
        #         my ($b, $e) = $self->ast_traverse($_, $jt);
        #         &connectNode($last_end, $b);
        #         $last_end = $e;
        #     }
        #     &connectNode($last_end, $end);
        #     push(@{$self->{begin_node}}, $begin);
        #     push(@{$self->{end_node}}, $end);
        # } else {
        #     $begin = $self->createNode("#global-begin", $el);
        #     $end = $self->createNode("#global-end", $el);
        #     &setNodePair($begin, $end);
        #     my $last_end = $begin;
        #     foreach (@children) {
        #         my ($b, $e) = $self->ast_traverse($_, $jump_tbl);
        #         next unless $b;
        #         if ($b->{label} =~ /^#func-/) {
        #             push(@{$self->{begin_node}}, $b);
        #             push(@{$self->{end_node}}, $e);
        #         } else {
        #             &connectNode($last_end, $b);
        #             $last_end = $e;
        #         }
        #     }
        #     &connectNode($last_end, $end);
        #     push(@{$self->{begin_node}}, $begin);
        #     push(@{$self->{end_node}}, $end);
        # }
    }

    return ($begin, $end, $jump_tbl);
}

sub jump_to {
    my $self = shift;
    my $tbl = shift;
    my $type = shift;

    if (!defined $tbl) {
        return undef;
    } elsif (exists $tbl->{$type}) {
        return $tbl->{$type};
    } else {
        my $top = $self->jump_to($tbl->{p}, $type);

        if (exists $tbl->{esc}) {
            my $esc_blk = $self->cloneNode($tbl->{esc});
            $esc_blk->{label} .= $type;
            &connectNode($esc_blk, $top);
            $top = $esc_blk;
        } elsif (exists $tbl->{blk}) {
            my $blk = $self->cloneNode($tbl->{blk});
            &connectNode($blk, $top);
            $top = $blk;
        }

        return $top;
    }
}

sub cleanup_switch_label {
    my $tbl = shift;
    if (exists $tbl->{esc} &&
        $tbl->{esc}->{label} =~ /^#(case|default)/) {
        return &cleanup_switch_label($tbl->{p});
    } else {
        return $tbl;
    }
}

sub switch_to {
    my $self = shift;
    my $tbl = shift;
    my $case = shift;
    my $el = shift;

    if (!defined $tbl) {
        return undef;
    } elsif ($tbl->{sw}) {
        my $case_node = $self->createNode($case, $el);
        &connectNode($tbl->{sw}, $case_node);
        return $case_node;
    } else {
        my $tail = $self->switch_to($tbl->{p}, $case, $el);
        if ($tbl->{blk}) {
            my $blk = $self->cloneNode($tbl->{blk});
            $blk->{label} =~ s/end$/begin/;
            &connectNode($tail, $blk);
            $tail = $blk;
        }
        return $tail;
    }
}

# goto 文により、外からラベル文に入るときの入口を作る
sub jump_in {
    my $self = shift;
    my $tbl = shift;

    if (!defined $tbl) {
        return (undef, undef); # may never happen.
    } elsif (exists $tbl->{out}) {
        my $node = $self->createNode("#label_entry", $tbl->{out}->ast_node);
        #        print STDERR "DEBUG: jump_in", $node->label, ":", $node->id, "\n";
        return ($node, $node);
    }

    my $blk;
    if (exists $tbl->{esc}) {
        $blk = $self->cloneNode($tbl->{esc});
        $blk->{label} .= 'go_in';
    } elsif (exists $tbl->{blk}) {
        $blk = $self->cloneNode($tbl->{blk}->pair_begin_one);
    } elsif (exists $tbl->{brk}) {
        $blk = $self->cloneNode($tbl->{brk}->pair_begin_one);
    } else {
        return (undef, undef);
    }
    $blk->{next} = [];
    $blk->{prev} = [];

    #    print STDERR "DEBUG: jump_in", $blk->label, ":", $blk->id, "\n";
    #    $Data::Dumper::Maxdepth = 2; print STDERR Dumper($blk);

    my ($top, $tail) = $self->jump_in($tbl->{p});

    unless ($top) { # $tail is also undef.
        $top = $tail = $blk;
    } else {
        #        print STDERR "DEBUG: jump_in connect ", $tail->label, ":", $tail->id,
        #            " => ", $blk->label, ":", $blk->id, "\n";
        &connectNode($tail, $blk);
        $tail = $blk;
    }

    return ($top, $tail);
}

# goto 文で外に出る経路を作る
sub jump_out {
    my $self = shift;
    my $tbl = shift;

    if (!defined $tbl) {
        return (undef, undef);
    }

    my $blk;
    if (exists $tbl->{esc}) {
        $blk = $self->cloneNode($tbl->{esc});
        $blk->{label} .= 'go_out';
    } elsif (exists $tbl->{blk}) {
        $blk = $self->cloneNode($tbl->{blk});
    } elsif (exists $tbl->{brk}) {
        $blk = $self->cloneNode($tbl->{brk});
    } else {
        return (undef, undef);
    }
    return (undef, undef) if $blk->label eq "#unit-end";
    $blk->{next} = [];
    $blk->{prev} = [];

    #    print STDERR "DEBUG: jump_out", $blk->label, ":", $blk->id, "\n";

    my ($top, $tail) = $self->jump_out($tbl->{p});

    unless ($top) {
        $top = $tail = $blk;
    } else {
        &connectNode($blk, $tail);
        $tail = $blk;
    }

    return ($top, $tail);
}

sub createNodeWithExpr {
    my $self = shift;
    my ($label, $str, $node) = @_;
    my ($begin, $end);
    if ($self->{opts}->{abst_level} eq "S") {
        $begin = $end = $self->createNode("$label:$str", $node);
    } else {
        $begin = $self->createNode("#$label-begin", $node);
        $end = $self->createNode("#$label-end", $node);
        &setNodePair($begin, $end);
        my @e = $self->ast_traverse_expr($node);
        &connectNode($begin, $e[0]);
        &connectNode($e[1], $end);
    }
    return ($begin, $end);

}

sub ast_traverse_decl {
    my $self = shift;
    my $el = shift;
    my $begin = $self->createNode("#decl-begin", $el);
    my $end = $self->createNode("#decl-end", $el);
    &setNodePair($begin, $end);
    my $tail = $begin;
    foreach my $d ($el->children("decr")) {
        my $decr_begin = $self->createNode("#decr-begin", $d);
        &connectNode($tail, $decr_begin);
        $tail = $self->createNode("#decr-end", $d);
        &setNodePair($decr_begin, $tail);

        my @left;
        if ($d->{t} eq "P" && $d->{sym} eq "_=_") {
            my ($var, $val) = $d->children("operand");

            @left = $self->ast_traverse_expr($var);

            my @right = $self->ast_traverse_expr($val);

            my $assign = $self->createNode("assign: " . $d->{sym}, $d);

            &connectNode($left[1], $right[0]);
            &connectNode($right[1], $assign);
            &connectNode($assign, $tail);
        } elsif ($d->{t} eq "P" || $d->{t} eq "ID_VF") {
            @left = $self->ast_traverse_expr($d);
            &connectNode($left[1], $tail);
        } # 構造体や enum などに対応していない
        if (@left) {
            &connectNode($decr_begin, $left[0]);

            # 宣言子の演算子を別のラベル名にして、通常の式と区別する
            my $n = $left[0];
            if ($self->{opts}->{with_expr_node}) {
                while ($n->{label} eq "#expr-begin") {
                    $n = $n->next_one;
                }
            }
            if ($n->{label} =~ s/^rvalue/d_value/) {
                $n = $n->next_one;
            }
            while ($n) {
                $n->{label} =~ s/^(op(\(lvalue\))?: (\*_|_\[))\b/d_$1/;
                last if ($n == $left[1]);
                $n = $n->next_one;
            }
        } else {
            &connectNode($decr_begin, $tail);
        }
    }
    &connectNode($tail, $end);
    return ($begin, $end);
}

sub ast_traverse_initlist {
    my $self = shift;
    my $el = shift;
    my ($begin, $end) = map($self->createNode("#initlist-$_", $el), "begin", "end");
    &setNodePair($begin, $end);
    my $cur = $begin;
    foreach ($el->children("list")) {
        my ($b, $e) = $self->ast_traverse_expr($_);
        &connectNode($cur, $b);
        $cur = $e;
    }
    &connectNode($cur, $end);
    return ($begin, $end);
}

sub ast_traverse_expr {
    my $self = shift;
    my $el = shift;
    my ($begin, $end);

    if ($el->{t} eq "CP") {
        ($begin, $end) = $self->ast_traverse_initlist($el);
        return ($begin, $end);
    } elsif ($el->{t} eq "DE") {
        ($begin, $end) = $self->ast_traverse_decl($el);
        return ($begin, $end);
    }

    if ($el->{t} eq "P" && exists $el->{sym}) {
        # logical operator and parthenthes only
        my @operand = $el->children("operand");
        if ($el->{sym} =~ /^_(\|\||\&\&)_$/) {
            my $bool = $self->eval_literal_cond($operand[0]);

            my ($go_exit, $go_next) =
                ( [ $self->create_bool_node("#true", $operand[0], $bool) ],
                  [ $self->create_bool_node("#false", $operand[0], $bool) ]);

            if ($el->{sym} eq "_&&_") {
                ($go_exit, $go_next) = ($go_next, $go_exit)
            }

            my ($b, $e) = $self->ast_traverse_expr($operand[0]);
            $begin = $b;
	    if ($bool ne "T") {
            &connectNode($e, @$go_exit);
	    }
	    if ($bool ne "F") {
            &connectNode($e, @$go_next);
	    }
            ($b, $e) = $self->ast_traverse_expr($operand[1]);
            &connectNode($go_next->[-1], $b);

            $end = $self->createNode("logical_op:" . $el->{sym}, $el);
            push(@{$self->{logical_op}}, $end);  # saves them for removing later.
            &connectNode($e, $end);
            &connectNode($go_exit->[-1], $end);
            return ($begin, $end);
        } elsif ($el->{sym} eq "!_") {
            my ($b, $e) = $self->ast_traverse_expr($operand[0]);
            $begin = $b;
            $end = $self->createNode("logical_op:" . $el->{sym}, $el);
            &connectNode($e, $end);
            return ($begin, $end);
        } elsif ($el->{sym} eq "(_)") { # parrentheses (ignore)
            ($begin, $end) = $self->ast_traverse_expr($operand[0]);
            return ($begin, $end);
        }
    }
    if ($self->{opts}->{abst_level} eq "L") { # stopped here for logical expression level.
        $begin = $end = $self->createNode("expr:". $el->str, $el);
        return ($begin, $end);
    }

    if ($el->{t} eq "P") {
        if (exists $el->{call}) { # function call
            # 関数名の評価を先にやる場合
            # my $call = $el->child("call");
            # ($begin, $end) = $self->ast_traverse_expr($call);
            # my @args = $el->children("arg");
            # my $cur = $end;
            # foreach my $arg (@args) {
            #     my ($b, $e) = $self->ast_traverse_expr($arg);
            #     &connectNode($cur, $b);
            #     $cur = $e;
            # }
            # my $sym = "_(" . join(",", map("_", @args)). ")";
            # $end = $self->createNode("call: " . $sym, $el);
            # &connectNode($cur, $end);

            # 関数名の評価を後にする場合 (これの方が自然)
            my @args = $el->children("arg");
            $begin = my $cur = $self->createNode("#call_dummy", $el);
            foreach my $arg (@args) {
                my ($b, $e) = $self->ast_traverse_expr($arg);
                &connectNode($cur, $b);
                $cur = $e;
            }
            my $call = $el->child("call");
            my ($b, $e) = $self->ast_traverse_expr($call);
            &connectNode($cur, $b);
            my $sym = "_(" . join(",", map("_", @args)). ")";
            $end = $self->createNode("call: " . $sym, $el);
            &connectNode($e, $end);
            $begin = $begin->next_one;
        } else {
            my @operand = $el->children("operand");
            if (@operand == 0) { # empty expression
                $begin = $end = $self->createNode("#empty", $el);
            } elsif ($el->{sym} =~ /^_[-+*\/%&|]?=_$/) { #assignment
                my @left = $self->ast_traverse_expr($operand[0]);
                my @right = $self->ast_traverse_expr($operand[1]);

                $begin = $left[0];
                $end = $self->createNode("assign: " . $el->{sym}, $el);
                &connectNode($left[1], $right[0]);
                &connectNode($right[1], $end);
            } elsif ($el->{sym} =~ /^_?(\+\+|--)_?$/) { # inc, decl
                my ($b, $e) = $self->ast_traverse_expr($operand[0]);
                $begin = $b;
                $end = $self->createNode("assign: " . $el->{sym}, $el);
                &connectNode($e, $end);
            } elsif ($el->{sym} eq "_?_:_") { # tri-operator
                my ($b, $e) = $self->ast_traverse_expr($operand[0]);
                $begin = $b;
                $end = $self->createNode("tri_op:" . $el->{sym}, $el);

#                $Data::Dumper::Maxdepth = 2; print Dumper($operand[0]);
                my $bool = $self->eval_literal_cond($operand[0]);

                if ($bool ne "F") {
                    my ($b1, $e1) = $self->ast_traverse_expr($operand[1]);
                    &connectNode($e, $self->create_bool_node("#true", $operand[0], $bool), $b1);
                    &connectNode($e1, $end);
                }
                if ($bool ne "T") {
                    my ($b2, $e2) = $self->ast_traverse_expr($operand[2]);
                    &connectNode($e, $self->create_bool_node("#false", $operand[0], $bool), $b2);
                    &connectNode($e2, $end);
                }
            } else {
                my $cur;
                foreach my $expr (@operand) {
                    my ($b, $e) = $self->ast_traverse_expr($expr);
                    $begin = $b unless ($begin);
                    &connectNode($cur, $b) if $cur;
                    $cur = $e;
                }
                my $ref_type = (exists $el->{left} ?  "($el->{left}value)" : "");
                my $sym = $el->{sym};
                $sym = "($el->{stype})_" if ($sym eq "T_");
                $end = $self->createNode("op$ref_type: $sym", $el);
                &connectNode($cur, $end);
            }
        }
    } elsif ($el->{t} =~ /^LI/) {
        $begin = $end = $self->createNode("literal: " . $el->{value}, $el);
    } elsif ($el->{t} eq "ID_MB") {
        my $ref_type = (exists $el->{left} ? $el->{left} : "");
        $begin = $end = $self->createNode($ref_type . "member: " . $el->{name}, $el);
    } elsif ($el->{t} =~ /^ID/) {
        my $ref_type = (exists $el->{left} ? "$el->{left}value: " : "rvalue: ");
        $begin = $end = $self->createNode($ref_type . $el->{name}, $el);
    }

    if ($self->{opts}->{with_expr_node}) {
        my $b = $self->createNode("#expr-begin", $el);
        my $e = $self->createNode("#expr-end", $el);
        &setNodePair($b, $e);
        &connectNode($b, $begin);
        &connectNode($end, $e);
        ($begin, $end) = ($b, $e);
    }
    return ($begin, $end);
}

sub ast_traverse_args {
    my $self = shift;
    my @args = @_;
    my @node;
    foreach my $el (reverse @args) {
        my $org_el;
        while (1) {
            $org_el = $el;
            if ($el->{t} eq "DE") {
                my @c = $el->children('decr');
               unless (@c) {  # empty declaration, that means no argument.
                    return (undef, undef);
                }

                $el = pop(@c);
            } elsif ($el->{t} eq "P") {
                if (exists $el->{call}) {
                    ($el) = $el->children('call');
                } else {
                    ($el) = $el->children('operand');
                }
            } elsif ($el->{t} eq "ID_VF") {
                push(@node, $self->createNode("arg: " . $el->{name}, $el));
                last;
            }
            if ($org_el eq $el) {
                die "ast_traverse_args: can't find ID_VF.\n" . Dumper($el);
            }
            last unless $el;
        }
    }
    &connectNode(@node);
    return ($node[0], $node[-1]);
}


sub createNode {
    my $self = shift;
    return CFG::NODE->new(++$self->{_node_id}, @_);
}

sub cloneNode {
    my $self = shift;
    my $node = shift;
    return $node->clone($self);
}

sub connectNode
{
    my $src = shift;

    foreach my $dst (@_) {
        $src->connect($dst) if $dst && $src;
        $src = $dst;
    }
}

sub setNodePair
{
    my ($begin, $end) = @_;

    return unless $begin && $end;
    $begin->add_pair_end($end);
    $end->add_pair_begin($begin);
}


sub createLabelNode
{
    my $self = shift;
    my ($name, $el, $jump_tbl) = @_;
    my $n = $self->createNode($name, $el);
    my ($top, $tail) = $self->jump_in($jump_tbl);
    &connectNode($tail, $n);
    $self->{_label_node}->{$el->{id}} = $top;
    return $n;
}

sub saveJumpNode
{
    my $self = shift;
    my ($node, $jump_tbl) = @_;
    my ($top, $tail) = $self->jump_out($jump_tbl);
    &connectNode($node, $tail);
    push(@{$self->{_jump_node}}, [ $node, $top ]);
}

sub connectJumpNode
{
    my $self = shift;
    foreach my $src (@{$self->{_jump_node}}) {
        my $dst_id = $src->[0]->ast_node->attr('jump_to');
        if (exists $self->{_label_node}->{$dst_id}) {
            &connectNode($src->[1], $self->{_label_node}->{$dst_id});
        }
    }

    # removing redundant nodes on the paths between gotoes and labels.
    foreach my $cur (values %{$self->{_label_node}}) {
        # at first, removes the label nodes and then recconect the path.
        next unless $cur->prev;
        push(@{$cur->next_one->{prev}}, $cur->prev);
        foreach ($cur->prev) {
            $_->{next} = [ $cur->next_one ];
            &reconnect_jump_path($_, $cur->next_one);
        }
    }
}

sub reconnect_jump_path {
    my ($src, $dst) = @_;
    #    $Data::Dumper::Maxdepth = 2; print STDERR Dumper($src), Dumper($dst);
    # if both two nodes have the sanme ast node, they are the pair nodes
    # labeled as *-begin and *-end. if they connect each other, they are
    # redundant nodes on the path and to be removed.
    while ($src->ast_node == $dst->ast_node) {
        my $src_prev = $src->prev_one;
        my $dst_next = $dst->next_one;
        #        $Data::Dumper::Maxdepth = 3; print STDERR "DEBUG: before\n", Dumper($src_prev), Dumper($dst_next);
        $src_prev->{next} = $dst->{next};
        push(@{$dst_next->{prev}}, @{$src->{prev}});
        # The dst_next may have the previous nodes connecting other paths.
        #        $Data::Dumper::Maxdepth = 3; print STDERR "DEBUG: after\n", Dumper($src_prev), Dumper($dst_next);
        ($src, $dst) = ($src_prev ,$dst_next);
    }
}

# 条件式が定数の場合に恒真または恒偽であるかを判定して返す
sub eval_literal_cond {
    my $self = shift;
    my $c = shift;
    my $bool = "N";
    if ($self->{opts}->{literal_cond} && $c->{t} eq "LIN") {
	$bool = ($c->{value} != 0) ? "T" : "F";
    }
    return $bool;
}

sub create_bool_node {
    my $self = shift;
    my $cond = shift;  # 'true' or 'false';
    my $elem = shift;
    my $bool = shift;  # 'T' or 'F';

    push(my @node, $self->createNode($cond, $elem));
    if ($bool eq "T") {
	push(@node, $self->createNode("#always_true", $elem));
    } elsif ($bool eq "F") {
	push(@node, $self->createNode("#always_false", $elem));
    }
    return @node;
}


sub optimize_cfg
{
    my $self = shift;

    # visit forward all nodes.
    my %visit_f;
    my @nodes = @{$self->{begin_node}};
    while (my $n = shift @nodes) {
        next if $visit_f{$n};
        $visit_f{$n} = $n;
        push(@nodes, grep(!$visit_f{$_}, $n->next));
    }

    # visit backwords all nodes while removing unreachable nodes
    # and merging duplicated pathes.
    my %visit_r;
    @nodes = @{$self->{end_node}};
    while (my $n = shift @nodes) {
        next if $visit_r{$n};
        my @prev = grep($visit_f{$_}, $n->prev); # remove unrechables
        #        $n->{prev} = [ &merged_nodes(@prev) ]; # merge previous nodes

        # Keep only bidirectional links.
        my @d_prev = ();
        foreach my $p (@prev) {
            my $find = 0;
            foreach ($p->next) {
                if ($n == $_) {
                    $find = 1;
                    last;
                }
            }
            push(@d_prev, $p) if $find;
        }

        $n->{prev} = [ @d_prev ]; # Not merged
        # マージすると c_peval.pl で、visitor の合流に失敗する。
        $visit_r{$n} = $n;
        push(@nodes, grep(!$visit_r{$_}, $n->prev));
    }

    # remove unrechable pair_end nodes.
    # This happens when the end node is unrechable from the start node.
    # Ex. "L: x; goto L;'
    foreach (values %visit_f) {
        if (exists $_->{pair_end}) {
            $_->set_pair_end(grep($visit_r{$_}, $_->pair_end));
            if ($_->pair_end == 0) {
                delete $_->{pair_end};
            }
        }
    }

    # remove unrechable end_node;
    $self->{end_node} = [ grep($visit_f{$_}, @{$self->{end_node}}) ];
}

sub merged_nodes
{
    my @nodes = @_;
    return @nodes unless @nodes > 1;

    my $top = shift @nodes;
    my @res;

    foreach my $n (@nodes) {     # merge nodes to $top
        # Nodes should has same  ast nodes and next nodes.
        if ($top->label eq $n->label &&
            $top->ast_node == $n->ast_node  && &has_same_nexts($top, $n)) {
            foreach my $p ($n->prev) { # changes the link to $node to $top
                foreach my $p_next (@{$p->{next}}) {
                    $p_next = $top if $p_next == $n;
                }
            }
            # merge prev nodes from $n to $top
            my %prev;
            map($prev{$_} = $_, ($top->prev, $n->prev));
            $top->{prev} = [ values %prev ];
        } else {
            push(@res, $n);  # Not merged, keep it.
        }
    }
    return ($top, &merged_nodes(@res));
}

sub has_same_nexts
{
    return join(":", sort(map($_->{id}, $_[0]->next)))
        eq join(":", sort(map($_->{id}, $_[1]->next)));
}

# 論理演算子のノードを削除 (対象は && と || のみ。否定演算子 ! は含まない)
sub remove_logical_op {
    my $self = shift;
    foreach my $n (@{$self->{logical_op}}) {
        foreach my $p ($n->prev) { # 論理演算子の
            &reconnect_logical_path($p, $p);
        }

    }
}

# 再帰的に辿って、縮退可能なパスは繋ぎ直す
# ここでは論理演算子のノードは残るが、optimize_cfg で削除される。
sub reconnect_logical_path {
    my ($src, $node) = @_;

    my $src_label = $src->label;
    my $src_is_bool = $src_label =~ /^#(true|false)/;
    my @n = $node->next;
    $src->{next} = [];  # 注意: $src と $node が同じことがある
    foreach (@n) {
        if ( $src_is_bool && $_->label =~ /^#(true|false)/) {
            # 論理演算子によって生じる真偽が矛盾する経路の辺は無視する
            if ($src_label eq $_->label) { # 同じ条件のノードのパスを縮退
                $src->{next} = [ @{$_->{next}} ];
                push(@{$_->next_one->{prev}}, $src);
            }
        } elsif ($_->label =~ /^logical_op:_/) {
            # 論理演算子のノードならスキップして再帰的に辿る
            &reconnect_logical_path($src, $_);
        } else { # 省略できないノードに辿りついたので、そこまで縮退
            push(@{$src->{next}}, $_);
            push(@{$_->{prev}}, $src);
        }
    }
}

sub add_to_ast
{
    my $self = shift;
    my $cfg_node = [];
    $self->{ast}->{cfg} = {
        node => $cfg_node,
        begin => [ map($_->id, @{$self->{begin_node}}) ],
        end => [ map($_->id, @{$self->{end_node}}) ],
    };

    my %visit;
    my @b = @{$self->{begin_node}};
    while (my $n = shift @b) {
        next if $visit{$n->{id}};
        my @next = $n->next;
        my @prev = $n->prev;
        my $node = {
            id => $n->id,  # node id
            ast_id =>$n->ast_node->id,
            label => $n->label,
            next => [ map($_->id, @next) ],
            prev => [ map($_->id, @prev) ],
        };
        my @pair_begin = $n->pair_begin;
        my @pair_end = $n->pair_end;
        $node->{pair_begin} = [ map($_->id, @pair_begin) ] if @pair_begin;
        $node->{pair_end} = [ map($_->id, @pair_end) ] if @pair_end;

        push(@{$cfg_node}, $node);
        $visit{$n->id} = 1;
        push(@b, grep(!$visit{$_->id}, @next));
    }
}


#--------------------------------------------------------------------#

# CFG マップの生成 (CFG node id -> CFG node)
sub generate_cfg_node_map {
    my $ast = shift;           # AST
    my $cfg_node_map = {};     # CFG のノードマップ 空)

    # CFG の各ノードの定義(JSON から読み込んだもの)から、メモリ上の CFG のノードを生成
    # CFG の node map にも登録を行う。
    foreach my $n (@{$ast->{cfg}->{node}}) {
        my $node = {  # ノードの定義
            'id' => $n->{id}, 'label' => $n->{label},
                # 関係するAST のノードがあるなら、直接、そのノードへの参照を持つ
                'ast_node' => (defined $n->{ast_id} ?
                               $ast->node($n->{ast_id}) : undef),
                # CFG のノード定義への参照
                'cfg_node' => $n,
        };
        $node = bless $node, "CFG::NODE";
        $cfg_node_map->{$node->id} = $node; # マップに登録
    }
    # 注意: CFG のノードは、AST の中に組込まれていた CFG のノードリスト(ここでは
    # 「ノード定義と呼ぶ)と、このクラスで生成されるメモリ上の CFG を表現する
    #  ノードの2種類が存在する。対応する 2つのノードは、同じ id を持つ。

    # 生成したすべての CFG のノードについて、next と prev を追加
    foreach my $n (values %$cfg_node_map) {
        # CFG のノード定義の next, prev に id のリストがあるので、
        # CFG のノードに変換したリストを作り、それらを直接、参照する。
        $n->{next} = [ map($cfg_node_map->{$_}, @{$n->{cfg_node}->{next}}) ];
        $n->{prev} = [ map($cfg_node_map->{$_}, @{$n->{cfg_node}->{prev}}) ];

        $n->{pair_begin} = [ map($cfg_node_map->{$_}, @{$n->{cfg_node}->{pair_begin}}) ]
            if  exists $n->{cfg_node}->{pair_begin};
        $n->{pair_end} = [ map($cfg_node_map->{$_}, @{$n->{cfg_node}->{pair_end}}) ]
            if  exists $n->{cfg_node}->{pair_end};
    }

    return $cfg_node_map;  # 生成したマップを返す。
}

sub flush_cfg_node_map_to_json {
    my $self = shift;

    # CFG のノード定義リストを JSON 形式に変換して返す
    my $nodes = [];
    foreach my $n (values %{$self->{node_map}}) {
        my $cn = { %$n };
        $cn->{next} = [ map($_->{id}, @{$n->{next}}) ];
        $cn->{prev} = [ map($_->{id}, @{$n->{prev}}) ];

        $cn->{pair_begin} = [ map($_->{id}, @{$n->{pair_begin}}) ]
            if  exists $n->{pair_begin};
        $cn->{pair_end} = [ map($_->{id}, @{$n->{pair_end}}) ]
            if  exists $n->{pair_end};
        $cn->{ast_id} = $n->{ast_node}->id if $n->{ast_node};
        delete $cn->{ast_node}; # ast_node は不要
        delete $cn->{cfg_node}; # cfg_node は不要

        push(@$nodes, $cn);
    }
    $self->{ast}->{cfg}->{node} = $nodes;

    $self->{ast}->{cfg}->{begin} = [ map($_->id, $self->node_begin) ];
    $self->{ast}->{cfg}->{end} = [ map($_->id, $self->node_end) ];
}

# AST に親ノードへのリンクを追加 (JSON形式にするときは除去が必要)
sub add_parent_link_in_ast {
    my $self = shift;
    &_add_parent_link_in_ast($self->{ast});
    return $self;
}

sub _add_parent_link_in_ast {
    my $el = shift;        # 部分木の根

    foreach ($el->children_all) {  # 子要素について再帰的に登録
        $_->{parent} = $el;
        &_add_parent_link_in_ast($_);
    }
}

# AST の親ノードへのリンクを削除 (JSON形式にするときは除去が必要)
sub remove_parent_link_in_ast {
    my $self = shift;
    &_remove_parent_link_in_ast($self->{ast});
    return $self;
}

sub _remove_parent_link_in_ast {
    my $el = shift;        # 部分木の根

    foreach ($el->children_all) {  # 子要素について再帰的に削除
        delete $_->{parent};
        &_remove_parent_link_in_ast($_);
    }
}



#--------------------------------------------------------------------#
# CFG から GraphViz の dot 形式への変換

# CFG を dot 形式に変換して返す
sub to_dot {
    my $self = shift;

    # 単独 CFG のときはグレー系の色を採用する
    $self->{color}->{cfg} ||= { node => "#000000", font => "#000000",
                                edge => "#000000", be => "#F0F0F0",
                                pair_edge => "#606060" };
    # GraphViz のオブジェクトを用意
    $self->{graph} = GraphViz->new();

    # AST を GraphViz のオブジェクトに変換 (AST の色の指定があったときのみ)
    $self->ast2dot if exists $self->{color}->{ast};

    # CFG を GraphViz のオブジェクトに変換
    if ($self->{opts}->{as_shrinked_graph}) {
        $self->cfg2dot_s;
    } else {
        $self->cfg2dot;
    }

    # CFG の開始と終了のノードを目立たさせる
    foreach ( ($self->node_begin, $self->node_end) ) {
        $self->{graph}->add_node(&cfgid($_->{id}), style=>"filled",
                                 fillcolor => $self->{color}->{cfg}->{be});
    }

    # dot 形式で出力
    #    return $self->{graph}->as_text;
    my @out;
    foreach (split("\n", $self->{graph}->as_text)) {
        # GraphViz のエスケープ問題への対応 (cfg2dot);
        s/\\([<>])/$1/g; # 勝手にエスケープされているので、戻す。
        if (s/^(\s*label="\w+)_escaped:/$1:/) {
            s/\\/\\\\/g;
            s/&quot;/\\"/g;
            s/&amp;/&/g;
        }
        push(@out, $_);
    }
    return join("\n", @out), "\n";

}

# AST も同時に表示するオプション設定
sub with_ast_graph {
    my $self = shift;

    # CFG は赤系の色にする
    $self->{color}->{cfg} = { node => "#A00000", font => "#600000",
                              edge => "#800000", be => "#FFF0F0",
                              pair_edge => "#803030" };
    # AST は青系の色にする
    $self->{color}->{ast} = { node => "#0000A0", font => "#000060",
                              edge => "#000080", directive => "#F0F0FF"};
    # AST の色を設定すると、AST を表示するものとして処理される

    return $self;
}

sub as_shrinked_graph {
    my $self = shift;
    $self->{opts}->{as_shrinked_graph} = 1;
    return $self;
}


# AST を graphviz のオブジェクトに変換
sub ast2dot {
    my $self = shift;
    my $g = $self->{graph};            # graphviz のオブジェクト
    my $color = $self->{color}->{ast}; # AST の色設定

    # マップ内のすべてのノードについて、graphviz オブジェクトにノードを登録する
    foreach my $node (values %{$self->{ast}->{node_map}}) {
        # ノードの作成
        $g->add_node($node->id, label => &AST_Label($node), shape => "box",
                     color => $color->{node}, fontcolor => $color->{font});
        # ノードが directive のときは、形を変える
        if ($node->{t} =~/^DIRE_/) {
            $g->add_node($node->id, shape => "hexagon", style=>"filled",
                         color=> $color->{directive});
        }
        # 子要素への辺を作る
        #	foreach my $ch (grep(&isObj($_), @{$node->{e}})) {
        foreach my $ch ($node->children_all) {
            $g->add_edge($node->id, $ch->id, weight => 3,
                         color => $color->{edge});
        }
    }
}

# AST のノードのラベルの作成
sub AST_Label {
    my $el = shift;
    my $label = $el->{t};   # 基本はノードの種別名
    if ($label =~ /^ID_/) { # 識別子の場合は名前を付ける
        $label .= " : " . $el->{name};
    } elsif ($label =~ /^P/) { # 式の場合は演算と呼出しを区別する
        $label = "op: " . $el->{sym} if defined $el->{sym};
        $label = "call" if exists $el->{call};
    } if ($el->{t} =~ /^LI/) { # 定数(literal)の場合は値を付ける
        $label .= " : " . $el->{value};
    }

    # GraphViz.pm のダブルクォートの処理がおかしいので、補正
    $label =~ s/\\/\\\\/g;
    $label = &escape_for_GraphViz_bug($label);
    return $label;
}

# CFG を graphviz のオブジェクトに変換
sub cfg2dot {
    my $self = shift;
    my $g = $self->{graph};                      # graphviz のオブジェクト
    my $color = $self->{color}->{cfg};           # CFG の色設定
    my $with_ast = exists $self->{color}->{ast}; # AST も出力するかのフラグ

    my %pairs;  # ブロック構造(開始と終了)の組を記録する変数

    # マップ内のすべてのノードについて、graphviz オブジェクトにノードを登録する
    foreach my $node (values %{$self->{node_map}}) {
        # ノードの作成
        my $label = escape_for_GraphViz_bug($node->label);
        $g->add_node(&cfgid($node->id),
                     label => $label .":". $node->id,
                     color => $color->{node}, fontcolor => $color->{font});

        # next の関係の辺を作成
        foreach ($node->next) {
            $g->add_edge(&cfgid($node->id), &cfgid($_->id), weight => 4,
                         color => $color->{edge});
        }

        # *-begin と *-end を結ぶ辺
        if (exists $node->{pair_end}) {
            foreach ($node->pair_end) {
		next unless $_;
                $g->add_edge(&cfgid($node->id), &cfgid($_->id), weight => 1,
                             style => "dotted", color => $color->{pair_edge},
                             dir => "none");
            }
        }
        # AST を含める場合は、AST へのノードの辺を作成
        if ($with_ast && defined $node->{ast_node}) {
            $g->add_edge(&cfgid($node->id), $node->ast_node->id,
                         style => "dashed", weight => 1);
        }
    }
}

# CFG を graphviz のオブジェクトに変換 (縮小版)
sub cfg2dot_s {
    my $self = shift;
    my $g = $self->{graph};                      # graphviz のオブジェクト
    my $color = $self->{color}->{cfg};           # CFG の色設定
    my $with_ast = exists $self->{color}->{ast}; # AST も出力するかのフラグ

    my %pairs;  # ブロック構造(開始と終了)の組を記録する変数

    # マップ内のノードのうち、必要なものだけを graphviz オブジェクトにノードを登録する
    foreach my $node (values %{$self->{node_map}}) {

        # '#' で始まるノードは無視。ただし、'#unit-begin' と '#unit-end'、
        # '#func-begin', '#func-end' は残す
        # 論理演算子も省略
        next if ($node->label =~ /^#(?!unit|func)|logical_op:_/);

        # ノードの作成
        my $label = escape_for_GraphViz_bug($node->label);
        $g->add_node(&cfgid($node->id),
                     label => $label .":". $node->id,
                     color => $color->{node}, fontcolor => $color->{font});

        # next の関係の辺を作成
        &create_short_edge($g, $color, $node, $node);

        # AST を含める場合は、AST へのノードの辺を作成
        if ($with_ast && defined $node->{ast_node}) {
            $g->add_edge(&cfgid($node->id), $node->ast_node->id,
                         style => "dashed", weight => 1);
        }
    }
}

sub escape_for_GraphViz_bug {
    my $label = shift;
    if ($label =~ /['"]/) {
        # GraphViz.pm が dot に変換するときのエスケープ処理がおかしい。
        # 一度、HTML の表現方式に直して、最後に出力するときに戻すようにする。
        $label =~ s/^(\w+):/$1_escaped:/;
        $label =~ s/&/&amp;/g;
        $label =~ s/"/&quot;/g;
        #            print STDERR "DEBUG: $label\n" if $label =~ /%/;
    }
    return $label;
}


# 再帰的に辺を辿り、表示すべきノードとの間の辺を作る
sub create_short_edge {
    my ($g, $color, $src, $node, $label) = @_;

    foreach ($node->next) {
        my $next_label = $label;
        if ($_->label =~ /^#(true|false)/) { # 真偽は辺ラベルに追加
            # 論理演算子によって生じる真偽が矛盾する経路の辺は無視する
            next if ($label && $label ne $_->label);
            $next_label = $_->label; # 1つしかありえないので、代入のみ
        }
        if ($_->label =~ /^#(?!unit|func)|logical_op:_/) {
            # 表示すべきではないノードならスキップして再帰的に辿る
            &create_short_edge($g, $color, $src, $_, $next_label);
        } else {
            # 辺を作成
            $g->add_edge(&cfgid($src->id), &cfgid($_->id), weight => 4,
                         color => $color->{edge}, label => $label);
        }
    }
}


# CFG 用の ID の作成
sub cfgid { return "c".$_[0] };
# 補足: CFG と AST は、各ノードの id の割り当てを独立に行なっているので、
# id が重複する。GraphViz で両方を同時に構成する場合、GraphViz のノードの
# id は、それぞれ区別する必要があるので、CFG の方は接頭辞として "c" を
# 付けるようにしている。

1;

#########################################################################
# CFG のノードのクラス
package CFG::NODE;

sub new {
    my $class = shift;
    my ($id, $label, $node) = @_;
    #    $label =~ s/\"/\\"/g;  # GraphViz.pm has a bug escaping double quotes twice.
    my $n = {label => $label, ast_node => $node, id => $id };
    return bless $n;
}

sub clone {
    my $self = shift;
    my $cfg = shift;
    my $clone = bless { %$self };
    $clone->{id} = ++$cfg->{_node_id};

    # クローンされたら、pair_begin, pair_end にも反映
    if (exists $clone->{pair_begin}) {
        map( push(@{$_->{pair_end}}, $clone), @{$clone->{pair_begin}})
    }
    if (exists $clone->{pair_end}) {
        map( push(@{$_->{pair_begin}}, $clone), @{$clone->{pair_end}})
    }
    return $clone;
}

sub id {
    return $_[0]->{id};
}

sub label {
    return $_[0]->{label};
}

sub ast_node {
    return $_[0]->{ast_node};
}

sub next {
    return @{$_[0]->{next}||[]};
}

sub next_one {
    # どれか1つだけ返す
    return $_[0]->{next}->[0] || undef;
}

sub prev {
    return @{$_[0]->{prev}||[]};
}

sub prev_one {
    # どれか1つだけ返す
    return $_[0]->{prev}->[0] || undef;
}

sub pair_begin {
    return @{$_[0]->{pair_begin}||[]}
}

sub pair_begin_one {
    # 通常は 1 つしか存在しない。
    return $_[0]->{pair_begin}->[0] || undef;
}

sub set_pair_begin {
    my $self = shift;
    $self->{pair_begin} = [ @_];
}

sub add_pair_begin {
    my $self = shift;
    push(@{$self->{pair_begin}}, @_);
}

sub pair_end {
    return @{$_[0]->{pair_end}||[]}
}

sub set_pair_end {
    my $self = shift;
    $self->{pair_end} = [ @_];
}

sub add_pair_end {
    my $self = shift;
    push(@{$self->{pair_end}}, @_);
}


sub connect {
    my ($src, $dst) = @_;
    push(@{$src->{next}}, $dst);
    push(@{$dst->{prev}}, $src);
}




#########################################################################
# CFG 上を動く visitor クラス

package CFG::VISITOR;

# 以下の記述は perldoc CFG.pm と実行することで読める。
=pod

=encoding utf8

=head1 CFG::VISITOR の使い方

基本的な処理の流れは以下の通り。必要に応じて、visitor を拡張し、関係する初期化を行う。

=over

=item (1) visitor を生成する。


  $cfg = CFG->new($ast); # $cfg を生成しておき
  $visitor = $cfg->visitor; # visitor を生成する

=item (2) 移動方向を設定する。

  $visitor->set_direction_next; # 順方向の場合
  $visitor->set_direction_prev; # 逆方向の場合

=item (3) 各種設定をする

  (具体的な analyzer, clone_copy などは後述)

=item (4) visitor を出発すべきノードに設定する

  $visitor->move_to($node);  # $node は移動開始となる CFG のノード

=item  (5) visitor を動かす

  $visitor->run;

=back

(3) については、必要に応じて、以下の定義を行う。

=head2 (a) 分析処理

=head3 (方法1)

属性 analyzer に手続きを無名ルーチンとして登録する。
引数として、その visitor の参照が与えられる

  例: $visitor->{analyzer} = sub { my $v = shift; ... 解析処理  ... }

visitor はハッシュであり、visitor が参照するCFGのノードはキー cur で取得する。
CFG のノードもハッシュであり、次のキーを持つ

    label => ノードのラベル
    id => ノードの識別番号 (AST のノードとは独立)
    next => CFG の次のノードへの参照を持つリストの参照
    prev => CFG の前のノードへの参照を持つリストの参照
    ast_node => 対応するAST のノードの参照

例:
    my $v = shift; # visitor が代入されたとする。
    my $cfg_node = $v->{cur}; # visitor が参照している CFGノード
    my $ast_node = $cfg_node->{ast_node}; # 対応する AST ノード

=head3 (方法2)

ノードのラベルのパターンと処理の表を登録する。
表はハッシュ { p => パターン, a => 処理 } のリストとして作る。

例:

  my $tbl = [
    { p => "assign", a => sub { ... } },
    { p => qr/^op/, a => sub { ... } },
    ...
  ];
  $visitor->build($tbl);  # analyzer が構成される。

パターンは、文字列または正規表現(qr/abc.*z/)を記述する。
文字列の場合はラベルの先頭のノードの種類(型)またはラベル全体と一致すると
処理が実行される。なお、文字列はハッシュ表に格納され、
処理が呼出されるので、同じ文字列を複数回指定することはできない。
特に正規表現によるパターンと混在して使うときは、文字列が優先されるので注意すること。

正規表現の場合はラベル全体と適合した場合に処理が実行される。
正規表現は、一致するか繰り返し比較されるので、同じ正規表現を
複数回使用してもよい。表に定義した順に比較が行われる。

パターンに適合する前に処理を実行したい場合には、インデックスとして
begin を持つものを、パターンに適合した処理をすべて終えた後に
実行した場合には インデックスとして end を持つものを用意し、
以下のようにそれぞれ無名サブルーチンで処理を定義する。

  my $tbl = [
    { begin => sub { ... } },
    ...
    { p => ..., a => sub { ... } },
    ...
    { end => sub { ... } }
  ];


処理は無名サブルーチンとして記述する。実行時には、引数に visitor が与えられる。
パターンの場合は、その後ろにパターン内で指定したグループ(丸括弧で囲んだもの)
がリストとして渡される。

なお、パターンは無名配列にしてもよい。その場合、各要素ごとに、同じ処理が
適用される。配列内は、文字列と正規表現を混在させてもよい。

例:

  my $tbl = [
    { p => [ "assign", qr/^#then-in/, qr/^#else-out/ ],
      a => sub { ... } },
    ...
  ];


=head2 (b) コピー手続き

visitor のクローンが生成されるときに、visitor に拡張して登録した属性は
浅いコピー(shallow copy)をされる。つまり、参照しているオブジェクトは
共有される。深いコピー(deep copy)をしたい場合、つまり、各 visitor ごとに、
独立したオブジェクトを保持したい場合は、その属性名を set_deep_copy_keys() で
登録をする。

  例: $visitor->set_deep_copy_keys('indenendent_key');

深いコピーは Clone::clone で実現される。もし特殊なコピーをしたい場合や
コピー(またはクローン生成)にともなって特別な処理をしたい場合は
属性 clone_copy にコピー処理のサブルーチンを登録する。
なお、clone_copy を設定しても、上記の浅いコピーと深いコピーは
適用される。

  例: $visitor->{clone_copy} = sub { ($v, $c) = @_;
                                     $c->{hoge} = { %$v->{hoge} };
                                     $c->{copy_count}++;
                                   };

=head2 (c) 環境シグネチャ

visitor が訪問済みのノードごとに残す足跡となる文字列を返す無名ルーチンを
属性 env_sig に定義する。常に同じ値を返せば、各ノードを1回だけ訪問することになる。
ノード訪問時の計算環境を記録させれば、異なる環境状況で訪問したときは
そのノードで再度解析処理が行われる。

  例: $visitor->{env_sig} = sub { return "x"; }

=head2 (d) ループカウント

set_loop_count_max() で visitor がループを回る回数の上限を指定できる。
visitor は、ループの条件分岐でクローンが生成され、一つはループの中へ、
もう1つはループの外へ進む。ループの中に入るときに、この上限に達していたら
最終状態に移行する。

ループのカウントは、visitor がループ外から入るときにリセットされる。
よって、二重ループのときは、外側のループを回るたびに、内側のループに
ついてはリセットされるので、1つの visitor が内側のループ内を最大で
設定した最大値の2乗回回ることになる。また、各繰り返しごとに、ループを
抜ける visitor が生成されることにも注意する必要がある。

ループの回数を固定し、かつ、ループ内に入る場合とそうでない場合の
2種類に限定する場合は set_loop_count() で指定する。
こちらは、set_loop_count_max() と基本的には同じだが、
最大の繰り返し回数に達する前に抜ける visitor は、そのときに
最終状態に移行する。

=head2 (e) visitor の id

visitor は各分岐に遭遇すと、clone を生成する。デバッグをするときは、
各 visitor を区別して状態を確認する必要があるので、visitor には
id を割り振っており、サブルーチン id で文字列化しものを取り出せる。

id は、1.1.1.2.2.1 のように、生成とクローンの履歴で表現される。
最初の数字は、new されるたびに1ずつ増加する値である。
そのあとは、分岐を進んだ本体なら 1 を、クローンされたものなら 2 が
追加されていく。なお、本体が分岐のどちらに進むかは制御できない。

=head2 (f) 終了処理

visitor が終了するときに、実行すべき処理があれば、属性 final に
無名ルーチンとして登録する。
例: $visitor->{final} = sub { print "Good bye\n"; }

=head1 その他

具体的な使用例は program_slicer.pl を参考にするとよい。

=cut

use Clone;
use Data::Dumper; #for debug

my $_visitor_id;

# visitor の生成
sub new {
    my $self = bless {};  # ハッシュとして生成
    $self->{cfg} = $_[1]; # visitor が走査する CFG
    $self->{queue} = [];  # 動作待ちの visitor のキュー
    $self->{visit} = {};  # 訪問先を記録するマップ

    $self->{block_in} = "begin";  # block に入るときのノードラベルの接尾語
    $self->{block_out} = "end";   # block を出るときのノードラベルの接尾語
    $self->{loop_in} = "#loop-in";  # block に入るときのノードラベルの接尾語
    $self->{loop_back} = "#loop-back";   # block を出るときのノードラベルの接尾語

    $self->{call_stack} = CFG::STACK->new; # 呼出し元に戻るためのスタック

    # visitor を区別するための id
    $self->{id} = [ ++$_visitor_id ];

    return $self;
}

# visitor の id の文字列表現
sub id {
    my $self = shift;
    return join(".", @{$self->{id}});
}

# id が長くなりがちなので、短く表現する
sub short_id {
    my $self = shift;
    my $prev = '';
    my $count = 0;
    my @res;
    foreach (@{$self->{id}}) {
        if ($prev ne $_) {
            if ($count > 0) {
                push(@res, "${prev}*${count}");
                $count = 0;
            }
        }
        $count++;
        $prev = $_;
    }
    if ($count > 0) {
        push(@res, "${prev}*${count}");
    }

    return join("-", @res);
}

# visitor の移動向きの設定 (next or prev)
sub set_direction {
    my $self = shift;
    $self->{direction} = shift;
    die "illegal direction $self->{direction}."
        unless $self->{direction} =~ /^(next|prev)$/;
    if ($self->{direction} eq 'prev') {
        ($self->{block_in}, $self->{block_out})
            = ($self->{block_out}, $self->{block_in});
        ($self->{loop_in}, $self->{loop_back})
            = ($self->{loop_back}, $self->{loop_in});
    }
    return $self;
}

# visitor の移動向きを next にする
sub set_direction_next {
    my $self = shift;
    return $self->set_direction('next');;
}

# visitor の移動向きを prev にする
sub set_direction_prev {
    my $self = shift;
    return $self->set_direction('prev');;
}

# visitor が現在参照しているノードを返す
sub cur {
    return $_[0]->{cur};
}

# visitor を引数のノードに移動させる
sub move_to {
    my $self = shift;
    $self->{cur} = shift;  # current (今、いるノード)

    # 経路に関する情報の記録(入口は in のみ、出口は 4種類)
    if ($self->{cur}->label =~ /^#(\w+)-(?:(in)|(out|back|brk|cnt))$/) {
        $self->save_path_info($1, $2, $3);
    }

    # ループの繰り返し回数の制御 (loop_count_max が設定されたときのみ)
    if (exists $self->{loop_count_max}) {
        if ($self->{cur}->label =~ /^#(while|for|do)-$self->{block_in}/) {
            push(@{$self->{loop_stack}}, 0); # reset the count for this loop.
        } elsif ($self->{cur}->label =~ /^#(while|for|do)-$self->{block_out}/) {
            my $c = pop(@{$self->{loop_stack}});
            if (exists $self->{loop_cout_out_cond}) {
                unless (&{$self->{loop_cout_out_cond}}($c)) { # 繰り返しの途中の脱出を許さないもの
                    $self->move_to_final_state;
                    return undef;
                }
            }
        } elsif ($self->{cur}->label eq $self->{loop_in}) { # 繰り返しに再突入
            my $c = ++$self->{loop_stack}->[-1];
	    if ($self->{cur}->prev_one->label eq "#always_true") {
		--$c;
	    }
            if ($c > $self->{loop_count_max}) { # 繰り返し回数が上限を越えたら終了
                $self->move_to_final_state;
                return undef;
            }
        }
    }

    return $self;
}

# visitor を引数のノードに移動させ、終了後に元のノードに戻る。
sub call_to {
    my $self = shift;
    my $call_node = shift;
    $self->{call_stack}->push( $self->{cur} );
    $self->{step_next_node} = $call_node; # 次の step で呼出し先に移動する
}

# visitor を1歩進める
sub step {
    my $self = shift;

    my @next;
    if ($self->{step_next_node}) { # 次の移動先が指定されているなら、それだけにする
        @next = ($self->{step_next_node});
        delete $self->{step_next_node};
    } else { # 移動方向にあるノードのリストを作成
        @next = @{$self->{cur}->{$self->{direction}} || []};
    }
    # すでに訪問済みのノードは除外
    @next = grep(!$self->already_visited($_), @next);


    # ToDo:
    # if ($self->is_suspending) {
    # 	# 停止中だったら、次の #true に進めるやつだけ進める
    # 	@next = grep($_->label eq '#true', @next;
    # 	$self->clone->move_to($next[0]);
    # 	return 0;
    # } elsif ($self->{just_after_resumed}) {
    # 	# 復帰した場合、#true には進まない。
    # 	# 復帰した場合とは、suspend で is_resumed が 0 にされたとき
    # 	# step の前の analysis の中で resume されることが前提
    # 	# resume したあとは、統合されて1つになっていることが前提
    # 	@next = grep($_->label ne '#true', @next);
    # 	delete $self->{just_after_resumed};
    # }


    # 訪問すべきノードがないなら、visitor を最終状態にする
    unless (@next) {
        if (@{$self->{call_stack}}) {
            my $cur = $self->{call_stack}->pop;
            $self->move_to($cur); # 呼出し元(caller)へ移動
            return $self->step; # 呼出し元(caller)の next へ移動
        } else { # どこからも呼出されていないなら
            $self->move_to_final_state;
            return -1;
        }
    }

    # 次に進むノード
    my $n = shift @next;

    # 訪問すべきノードが残っているなら、クローンを作って、それらを移動させる
    # move_to の副作用を回避するために、先にクローンを作成して移動
    foreach ( $self->clones(int(@next)) ) {
        $_->move_to(shift @next);
    }
#    print STDERR "#queue = ", int(@{$self->{queue}}), "\n";

    # 補足: クローンは queue の中で待機

    $self->move_to($n); # 次のノードへ進む

    return 0;
}

# visitor の clone を生成する
# sub Xclone {
#     my $self = shift;
#     my $clone = bless { %$self }; # 属性値のコピーを持つクローンを生成
#     # ここでの copy は浅い(shallow)ので、visitor に属性を追加している場合、
#     # その属性値を正しくコピーしているとは限らない。
#     # 追加した属性に関するコピー方法は、必要があれば、clone_copy に無名ルーチンとして
#     # 定義しておく。

#     # visitor の id の設定
#     $clone->{id} = [ @{$self->{id}}, 2 ]; # クローンには 2 を追加した新しいリスト
#     push(@{$self->{id}}, 1);              # 本体には 1 を追加
#     # Todo: 分岐が複数存在したときに問題あり
#     #       PDG::VISITORのように step で clone を作るときに id を増やすべき。

#     # 指定されたキーについて deep copy を行う。(loop_stack を含む)
#     foreach (@{$self->{deep_copy_keys}||[]}, 'loop_stack') {
#         $clone->{$_} = Clone::clone($self->{$_});
#     }

#     if (defined $self->{clone_copy}) {  # その定義があれば実行する。
#         &{$self->{clone_copy}}($self, $clone);
#     }

#     # クローンはすぐには動かないので、queue に入れておく。
#     push(@{$self->{queue}}, $clone);
#     return $clone;  # 生成したクローンを返す
# }

# visitor の clone を生成する
sub clone {  # 1つだけ生成する
    my $self = shift;
    my ($c) = $self->clones(1);
    return $c;
}

sub clones { # 指定の数だけ生成する。分岐数が多いときに利用。
    my $self = shift;
    my $num = shift; # 生成するクローンの数

    return () if $num < 1; # クローンが不要なら終了

    my @all;
    foreach my $i (2..$num+1) {
        my $clone = bless { %$self }; # 属性値のコピーを持つクローンを生成
        # ここでの copy は浅い(shallow)ので、visitor に属性を追加している場合、
        # その属性値を正しくコピーしているとは限らない。
        # 追加した属性に関するコピー方法は、必要があれば、clone_copy に無名ルーチンとして
        # 定義しておく。

        # visitor の id の設定
        $clone->{id} = [ @{$self->{id}}, $i ]; # クローンには 2 を追加した新しいリスト

        # 指定されたキーについて deep copy を行う。(loop_stack を含む)
        foreach (@{$self->{deep_copy_keys}||[]}, 'loop_stack') {
            $clone->{$_} = Clone::clone($self->{$_});
        }

        if (defined $self->{clone_copy}) {  # その定義があれば実行する。
            &{$self->{clone_copy}}($self, $clone);
        }

        # クローンはすぐには動かないので、queue に入れておく。
        push(@{$self->{queue}}, $clone);

        push(@all, $clone);
    }

    push(@{$self->{id}}, 1);              # 本体には 1 を追加し、他2以上にする。

    return @all;  # 生成したクローンを返す
}


sub set_deep_copy_keys {
    my $self = shift;
    $self->{deep_copy_keys} = [ @_ ];
    return $self;
}

sub add_deep_copy_keys {
    my $self = shift;
    push(@{$self->{deep_copy_keys}}, @_);
    return $self;
}


# visitor を動かす
sub run {
    my $self = shift;
    while ($self) {  # visitor が存在する限り、動き続ける
        if ($self->is_final || $self->is_suspending) { # visitor が最終状態または待機なら、
            #	    $self = shift @{$self->{queue}}; # queue のクローンと交代する
            #	    print STDERR "Unqueue ", ($self->is_suspending ? "suspending" : "-") , "\n";
            $self = $self->unqueue; # queue のクローンと交代する
            next;  # 交代すべきクローンがない場合があるので、while の条件式に戻る
        }
        $self->analyze;    # visitor が解析処理をする
        $self->mark_visited; # 現在のノードの訪問記録を付ける
        next if $self->is_suspending; # analyze の中で suspend されたとき
        unless ($self->is_final) {
            $self->step;         # visitor を1歩進める
        }
    }
}

# visitor が最終状態かどうを判定する
sub is_final {
    return $_[0]->{is_final_state};  # マークの有無で判定
}

sub continue {
    $_[0]->{is_final_state} = 0;
    $_[0]->run;
}

# visitor を最終状態に移行する
sub move_to_final_state {
    my $self = shift;
    $self->{is_final_state} = 1;  # 最終状態と記録する
    delete $self->{is_suspending};  # suspend の状態からは削除

    # 最終状態になるときに後始末が必要なら実行する
    if (defined $self->{final}) {
        return &{$self->{final}}($self);
    }
}

sub is_suspending {
    return $_[0]->{is_suspending};
}

sub suspend {
    my $self = shift;
    $self->{cur}->{suspend_queue} ||= [];
    my $sq = $self->{cur}->{suspend_queue};

    #    print STDERR "SUSPEND called!\n";
    #もし待ち状態を解消する条件が揃っていたら resume する。
    if ($self->resumable(@$sq)) {
        #	print STDERR "Resumed.\n";
        $self->resume(@$sq);
        $self->{cur}->{suspend_queue} = [];
    } else {
        #	print STDERR "Suspended.\n";
#        print STDERR "suspend: ", join(", ", $self->id, $self->label), "\n";
        $self->{is_suspending} = 1;
        push(@{$self->{queue}}, $self);  # 動作待ちキューに追加
        push(@$sq, $self); # そのノードの待ちキューに追加
    }
    #    print STDERR "  label: ", $self->label, "\n";
}

sub resumable {
    my $self = shift;
    unless (exists $self->{resumable}) {
        die "No resumable checking method exists.";
    }
    return $self->{resumable}->($self, @_);
}


sub resume {
    my $self = shift;

    unless (exists $self->{resume}) {
        die "No resume method exists.";
    }
    return $self->{resume}->($self, @_);

    # map($_->{is_suspending} = 0, ($self, @_));

    # 次の step で余分に進まないようにするためのフラグ
    # $self->{just_after_resumed};
}

sub unqueue {
    my $self = shift;
    my $q = $self->{queue};
    return undef unless @$q; # キューが空の場合
    #    print STDERR "Length of QUEUE: ", int(@$q), "\n";
    my $i;
    for ($i = 0; $i < @$q; $i++) {
        unless ($q->[$i]->is_suspending) {
            last;
        }
        #	print STDERR "$i in QUEUE is suspending.\n";
    }
    if ($i == @$q) {
        $Data::Dumper::Maxdepth = 3;
#        print STDERR Dumper($q);
        die "Dead lock error: All visitors suspend for waiting others.";
    }
    my $n = splice(@$q, $i, 1);
    return $n;
}


# queue で待機中の visitor を破棄する
sub kill_queue {
    my $self = shift;
    $self->{queue} = [ ];
}

# visitor が分析処理を行う。処理内容は analyzer 属性に定義
sub analyze {
    my $self = shift;
    $self->{analyzer}->($self) if $self->{analyzer};
}

# visitor が現在のノードを訪問済みかどうかを判定
sub already_visited {
    my $self = shift;
    my $node = shift;

    # signature が定義されていなければ常に偽を返す
    return 0 unless defined $self->{env_sig};

    # 判定は、signature が一致するかどうかで行う
    return (defined $self->{visited}->{$node}
            && $self->{visited}->{$node} eq $self->env_sig);
    # 注意: すでに訪問していても、signature が一致しなければ
    # 訪問していないという扱いになる。
}

# visitor が現在のノードを訪問済みと記録
sub mark_visited {
    my $self = shift;

    $self->{visited}->{$self->{cur}} = $self->env_sig;
}

# visitor が訪問時に使用する signature を返す (environment signature)
# 具体的な手続きは属性 env_sig に無名ルーチンとして登録する。
sub env_sig {
    my $self = shift;
    if (defined $self->{env_sig}) {
        return $self->{env_sig}->($self);
    } else {
        return undef;
    }
}

# visitor がループを回るときの回数の上限
# 上限回数を越えてループに入った visitor は、その時点で最終状態となり、終了する。
sub set_loop_count_max {
    my ($self, $max, $loop_cout_out_cond) = @_;

    $self->{loop_count_max} = $max;

    # $except_cond を設定すると、その条件を満たさないものは、ループから脱出できない。
    if ($loop_cout_out_cond) {
        die "extra condition of set_loop_count_max must be a anonymous function."
            unless ref($loop_cout_out_cond) eq "CODE";

        $self->{loop_cout_out_cond} = $loop_cout_out_cond;
    }
}

# visitor がループから出る条件を限定する例
# visitor は 0 回(ループ内に入らない)ものと、ループ内を指定回数だけ回るものだけになる。
# ループの実行回数が1回以上でかつ上限未満のものは、ループから抜けるときに終了する。
sub set_loop_count {
    my $self = shift;
    $self->{exact_loop_count} = shift;
    $self->set_loop_count_max($self->{exact_loop_count},
                              sub { ($_[0] == 0 || $_[0] == $self->{exact_loop_count}) });
    return $self;
}



# visitor の経路の情報を管理
# 順方向の場合は loop-in, loop_end-in, then-in, case_?-in, default-in
# 逆方向の場合は loop-{back,brk,cnt}, loop_end-out, then-{out,brk,cnt}, case_?-{out,brk,cnt}, default-{out,brk,cnt}
sub save_path_info {
    my $self = shift;
    my ($path, $in, $out) = @_;

    if ($self->{direction} eq "prev") {
        ($in, $out) = ($out, $in);
    }

    if ($in) {
        push(@{$self->{path_stack}}, "$path-$in");
    } else {
        $self->{path_last} = pop(@{$self->{path_stack}});
    }
}

# 現時点の経路
sub path_current {
    my $self = shift;
    return $self->{path_sktack}->[0] || undef;
}

# 最後の経路 (制御文の条件式にどの経路から入ったかを表す)
sub path_last {
    my $self = shift;
    return $self->{path_last} || undef;
}

# ASTのノードの wrapper (AST::NODE) を返す
sub ast_node {
    return $_[0]->{cur}->ast_node;
}

# visitor が訪問中のCFGノードのラベルを返す
sub label {
    my $self = shift;
    return $self->{cur}->label;
}

# visitor が訪問中のCFGノードのラベルの型名のみを返す
sub label_type {
    my $self = shift;
    my ($type) = ($self->{cur}->label =~ /^([^:]*)/);

    return $type;
}

# visitor が訪問中のCFGノードの id を返す
sub node_id {
    my $self = shift;
    return $self->{cur}->id;
}

# visitor の解析器を表から構築
sub build {
    my $self = shift;
    my $tbl = shift;
    # ラベルのパターンとそれに対する操作(無名サブルーチン)のリスト
    #  [ { p => "assign", a => sub { ... } }, ... ]
    # 無名サブルーチンの引数はビジターとラベル
    # ラベルのパターンは文字列または正規表現(qr/.../)

    my $t = { reg => [], str => {} };
    my ($begin, $end);
    foreach my $e (@$tbl) {
        if (exists $e->{begin}) {
            $begin = $e->{begin};
        } elsif (exists $e->{end}) {
            $end = $e->{end};
        } elsif (exists $e->{p}) {
            my @p = ref($e->{p}) eq "ARRAY" ? @{$e->{p}} : ($e->{p});

            foreach my $p (@p) {
                if (ref($p) eq "Regexp") {
                    push(@{$t->{reg}}, { p => $p, a => $e->{a} });
                } else {
                    $t->{str}->{$p} = $e->{a}
                }
            }
        }
    }

    $self->{analyzer} = sub {
        my $v = shift; # visitor
        my $str_tbl = $t->{str};
        my $type = $v->label_type;
        my $label = $v->label;

        $begin->($v) if ($begin);

        if (exists $str_tbl->{$type}) {
            $str_tbl->{$type}->($v, $type);
        } elsif (exists $str_tbl->{$label}) {
            $str_tbl->{$label}->($v, $label);
        } else {
            foreach my $e (@{$t->{reg}}) {
                if (my @p = ($label =~ $e->{p})) {
                    $e->{a}->($v, @p);
                }
            }
        }
        $end->($v) if ($end);

    };  # end of sub
}

#### スタッククラス

package CFG::STACK;

sub new {
    return bless [];
}


sub push {
    my $self = shift;
    push(@$self, @_);
}

sub pop {
    my $self = shift;
    return pop(@$self);
}

sub top {
    return $_[0]->at(-1);
}

sub at {
    my $self = shift;
    my $pos = shift;
    return $self->[$pos];
}

sub list {
    my $self = shift;
    return reverse @$self;
}

1;
