# 
# Copyright (c) 2009-2025 The TEBA Project. All rights reserved.
# 
# Redistribution and use in source, with or without modification, are
# permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# Author: Atsuhi Yoshida

package CParserProc;

=pod

=encoding utf8

=head1 CParserProc とは

CParserProc は TEBA の cparse.pl を呼出すモジュールである。

TEBA の構文解析モジュール CParser.pm は RewriteTokens.pm (Ver.1) で
構成されている。RewriteTokens2.pm を使うツールの場合、CParser.pm を
使って構文解析をしようとすると package RewriteTokens が二重に定義され、
エラーとなる。よって、RewriteTokens2.pm と CParser.pm は共存できず、
RewriteTokens2.pm を使うツールでは構文解析ができなかった。

CParserProc は、cparser.pl をサブプロセスとして起動し、
そのサブプロセスで構文解析を行う。これにより、ツール内で RewriteTokens.pm が
読み込まれることはないので、二重定義の問題は回避され、RewriteTokens2.pm を
使うツールでも構文解析が実現できる。

なお、cparser.pl を呼出すので、実行パスに cparser.pl が存在しなければ
ならない。また、/usr/bin/env を使用しているので、これが存在しない場合は
動作しない。

=head1 CParserProc の使い方の例

=head2 字句列として出力を得るとき

CParserProc のインスタンスの parse メソッドを呼出す。
引数はソーステキストである。

  my @tokens = CParserProc->new->parse($src_text);

出力は、TEBA の標準形式の字句のリストになる。

=head2 JSON形式の構文木を得るとき

CParserProc のインスタンスを、as_json メソッドで、JSON 形式出力用に
切り替えてから、parse メソッドを呼出す。
引数はソーステキストである。

  my $json_text = CParserProc->new->as_json->parse($src_text);

出力は、JSON形式のテキストである。なお、標準で記号表が付与される。
記号表を削るオプションは(現時点では)未定義である。

=cut

use strict;
use warnings;

#use FindBin qw($Bin);
#use lib "$Bin/../TEBA";

#use Data::Dumper;
use IPC::Open2;

sub new {
    my $self = bless {};
    $self->{opt}->{with_symboltable} = 1;  # default
    $self->{driver} = "cparse.pl";
    return $self;
}

sub as_json {
    my $self = shift;
    $self->{opt}->{as_json} = 1;
    return $self;
}

sub build {
    my $self = shift;
    my $opt = $self->{opt};
    my @args;
    push(@args, "-j") if $opt->{as_json};
    push(@args, "-S") unless $opt->{with_symboltable};

    $self->{script} = [ "/usr/bin/env", $self->{driver}, @args ];
    return $self;
}

sub parse {
    my $self = shift;
    my $src = shift;
    $self->build;


    my $pid = open2(my $p_out, my $p_in, @{$self->{script}});

    print $p_in $src;
    close($p_in);

    my @res = <$p_out>;
    close($p_out);

    waitpid($pid, 0);

    if ($self->{opt}->{as_json}) {
	return join('', @res);
    } else {
	return @res;
    }
}

1;
