# 
# Copyright (c) 2009-2025 The TEBA Project. All rights reserved.
# 
# Redistribution and use in source, with or without modification, are
# permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# Author: Atsuhi Yoshida

use CFG;

package PDG;

=pod

=encoding utf8

=head1 パッケージ PDG


プログラム依存グラフ(Program Dependency Graph)の生成を行うパッケージである。
ポインタ(エイリアス)の解析は行なっていないので、ポインタの先への代入は
ポインタを持つ変数への参照と代入と近似する。ポインタを持つ変数かどうかの判定は、
型がわかる場合はポインタ型かどうか、そうでない場合は最左の変数としており、
正確なものではない。

なお、依存関係の解析にあたり、変数の「定義・使用」という用語を使う場合には
以降の「参照」を「使用」に、「代入」を「定義」と読み替えること。

=head2 PDG オブジェクトの生成

PDG オブジェクトは、new メソッドで生成する。このとき、引数にCFG オブジェクトを渡すと、
PDG が構成される。標準入力から AST を読み込み、PDG を生成するまでの例を以下に示す。

   my $ast = AST::NODE->build_from_json_text(join('', <>));
   my $cfg = CFG->new($ast);
   my $pdg = PDG->new($cfg);

なお、new に $cfg を渡さない場合には、build メソッドで構成する。

   my $pdg = PDG->new();
   $pdg->build($cfg);

=head2 関数呼出しの実引数に関する代入

関数呼出しの実引数がポインタの場合、関数呼出しにおいて、そのポインタの先が代入される
可能性がある。どの引数において代入が発生するかは関数によって異なるので、引数の参照・代入に
対する補助情報を与えることができる。

補助情報は、関数名をキーとしたハッシュが、引数の参照・代入情報のリストを持つデータ構造で
与える。例えば、sprintf(3) と strcat(3) に対して、次のように定義する。

  {
    'sprintf' => [ 'l', 'r' ],
    'strcat' => [ 'lr', 'r' ],
  }

ここで、参照・代入の情報は次のように表現する。

=over

=item 'l' : 代入 (左辺値の意味)
=item 'r' : 参照 (右辺値の意味)
=item 'lr' : 代入と参照 (元の値を参照して代入。元の値の一部を残しつつ、代入される場合も含む。)

=back

また、各情報は引数の先頭から対応させ、省略された引数については参照として扱う。
上記の例の場合、sprintf(3) は第1引数を代入、第2引数が参照、第3引数以降は省略されており、
それらを参照として扱うことを意味する。

=head2 パッケージ PDG のメソッド定義

=cut

# 良いプログラムを書くためのおまじない。
use warnings;
use strict;

use GraphViz;

use Data::Dumper;  # for debug

=pod

=head3 new: PDG オブジェクトの生成

PDG->new([ CFG ], [ 関数の参照代入定義 ] , [ オプション ] )

PDG オブジェクトを生成する。引数に CFG を与えると、内部で build メソッドにより
PDG を構成する。 合わせて、関数の参照代入定義のオブジェクト(ハッシュ)を渡すと、
解析を行う。オプションはデバッグ用に用いる。

=cut

# PDG オブジェクトの生成
sub new {
    my $class = shift;
    my $cfg = shift;    # AST with CFG
    my $fcall = shift || {};    # 関数呼出しの引数の代入参照の定義
    my $opts = shift || {};

    my $self = bless {}; # ハッシュとしてCFGオブジェクトを生成

    # 根が指定されていたら、PDG を生成
    $self->build($cfg, $fcall, $opts) if $cfg;
    return $self;
}

=pod

=head3 build: PDG オブジェクトの生成

$pdg->build( CFG, 関数の参照代入定義, オプション)

与えられた CFG および関数参照代入定義に基づいて PDG を生成し、保持する。
CFG::VISITOR を用いて解析する。

生成されるPDGは、PDG オブジェクトのキー cfg に格納されている CFG が持つ
AST に保存される。この PDG のノードは式要素の識別番号であり、そのまま
JSON 化できる形式である。エッジは、ソースノードの式番号からデスティネーション
ノードの式番号のリストへのハッシュで表現される。

ノードとなる式は、代入を伴う式、および式文や条件文などの最外の式である。
よって、内部に代入が含まれる式は、複数のノードで表現される。

=cut

sub build {
    my $self = shift;
    my $cfg = shift;
    my $fcall = shift;
    my $opts = shift;

    $self->{cfg} = $cfg;

    # CFG 上を動く visitor を作成。上から下に移動するよう設定。
    my $visitor = $cfg->visitor->set_direction_next;

    # visitor はループを最大で2回まで回るように設定。
    # 依存解析の場合は、これで繰り返しを考慮した依存関係をすべて網羅できる。
    $visitor->set_loop_count_max(2);

    # 依存解析に使うスタックやハッシュ表
    $visitor->{assign_stack} = CFG::STACK->new; # 計算のシミュレーション用スタック
    $visitor->{stmt} = CFG::STACK->new; # 解析中の文
    $visitor->{cond} = CFG::STACK->new; # 解析中の条件式
    $visitor->{var_assign} = {}; # 変数名 から その時点の変数の値に代入した式 を求めるハッシュ
    # 最適化のために visitor の合成を行う。よって変数に代入する式は複数あるものとして
    # 処理するので、この表は 変数名 => [ 式, 式, 式, ... ] になる。

    # # 論理演算子の式を分離したい... 論理演算子に到達する前のノードを確認するために覚える
    # $visitor->{last_cur} = undef; # 直前に通過したノード

    $visitor->{resumable} = \&resumable;
    $visitor->{resume} = \&resume;


    # これらの属性は、経路ごとに独立に使用する必要があるので、visitor ごとにスタックや
    # 変数の表の複製が作られるように deep copy の設定をする。
    $visitor->set_deep_copy_keys('assign_stack', 'stmt', 'cond', 'var_assign');

    # プログラム依存グラフ
    $visitor->{pdg} = { 'data' => {}, 'ctrl' => {}, 'e2s' => {}, 'func' => [] };
    # e2s は式から文へのマッピング用

    ##### 解析処理の表 ######
    # CFG 上を移動しなら p にあてはまるノードがあれば a に設定した処理を実行する。
    my $tbl = [
        {   # 各ノードでの解析の先頭で行う共通処理 (主にデバッグ用)
            begin => sub {
                return unless $opts->{d};
                my $v = shift;
                print STDERR "----\nVISITOR: id = ", $v->id, "\n";
                print STDERR "Node: ", $v->label, "(", $v->label_type, ")\n";
                print STDERR "stmt: [", join(", ", @{$v->{stmt}}), "]\n";
                print STDERR "cond: [", join(", ", @{$v->{cond}}), "]\n";
                # print STDERR "var_assign: {", join(", ", map("$_ => $v->{var_assign}->{$_}",
                #                                              keys %{$v->{var_assign}})), "}\n";
                print STDERR "var_assign: {", join(", ", map("$_ => "
                                                             . join(", ", @{$v->{var_assign}->{$_}}),
                                                             keys %{$v->{var_assign}})), "}\n";
                print STDERR "assign_stack: [", Dumper($v->{assign_stack}), " ]\n";

                print STDERR "\n";
            }
        },

        {
            p => qr/#func-(\w+)-begin/,
            a => sub {
                my $v = shift;
                $v->{func} = $v->ast_node->id; # 解析中の関数の id を記憶
                $v->{cond}->push( $v->ast_node->id ); # 関数の制御下にあることにする
                push(@{$v->{pdg}->{func}}, $v->{func}); # PDG に追加
            }
        },


        { # 式文, if文, while文, for文, return文, 宣言の始まり
            p => qr/^#(expr_st|if|while|for|return|decl|switch)-begin/,
            a => sub {
                my $v = shift;
                $v->{stmt}->push( $v->ast_node->id ); # 解析中の文(宣言)の id を記憶
            }
        },

        { # 式文 (式を伴う要素)
            p => "#expr_st-end",
            a => sub {
                my $v = shift;
                &record_dependencies_for_expr($v);
                $v->{stmt}->pop; # 文の解析が終了
            }
        },

        { # 宣言子の終り, 前進式 (式を伴う要素)
            p => [ "#decr-end", "#for_succ-end" ],
            a => sub {
                my $v = shift;
                &record_dependencies_for_expr($v);
            }
        },

        { # return の返り値
            p => "#return_val-end",
            a => sub {
                my $v = shift;
                &record_dependencies_for_return($v);
            }
        },

        { # for文の初期化式
            p => "#for_init-end",
            a => sub {
                my $v = shift;
                my $prev = $v->cur->prev_one;
                unless ($prev->label eq "#decl-end") { # 直前が #decl-end なら #decr-end で処理されている
                    &record_dependencies_for_expr($v);
                }
            }
        },


        { # return文, 宣言の終り
            p => qr/^#(return|decl)-end/,
            a => sub {
                my $v = shift;
                $v->{stmt}->pop; # 文(宣言)の解析が終了
            }
        },

        {
            p => qr/^#(if|while|for|switch)-end$/, # 条件文の終了
            a => sub {
                my $v = shift;
                $v->{stmt}->pop; # 文(宣言)の解析が終了
                $v->{cond}->pop; # 制御依存の終了
            }
        },

        {
            p => "#cond-end", # 条件式の終了時
            a => sub {
                my $v = shift;
                &record_dependencies_for_expr($v);
                $v->{cond}->push( $v->ast_node->id ); # この条件式がこのあとの制御依存に加わる
            }
        },

        {
            p => "#loop-back", # ループの戻り
            a => sub {
                my $v = shift;
                $v->{cond}->pop; # 制御の条件になっている条件式を破棄
            }
        },

        {
            p => [ "literal", "member" ], # 定数値, 構造体のメンバ
            a => sub {
                my $v = shift;
                $v->{assign_stack}->push( [ ] ); # 参照する変数はないので、空リストを push
            }
        },

        {
            p => [ "lvalue", "rvalue" ], # 変数の参照
            a => sub {
                my $v = shift;
                my $value = { node => $v->ast_node,
                              ctx => $v->ast_node->left_ctx };
                $v->{assign_stack}->push( [ $value ] ); # この変数参照のリストが計算結果
                # リストにしているのは、近似的に複数の代入が起こる可能性に対応するため
            }
        },
        {
            p => "lrvalue", # 変数の参照
            a => sub {
                my $v = shift;
                my $value = { node => $v->ast_node,
                              ctx => $v->ast_node->left_ctx };
                $v->{assign_stack}->push( [ $value ] ); # 左辺値
                $v->{assign_stack}->push( [ $value ] ); # 右辺値
            }
        },

        {
            p => "op", # 代入以外の演算子
            a => sub {
                my $v = shift;
                # 被演算子を取り出し、それをマージしたもの結果とする
                my $num = int(@{$v->ast_node->{operand}});

                if ($num == 1) {
                    my $as_expr = $v->{assign_stack}->pop;
                    $v->{assign_stack}->push($as_expr);
                    # 結局、何もしなくてよい?
                } elsif ($num == 2) {
                    my @as_exprs = ($v->{assign_stack}->pop, $v->{assign_stack}->pop);
                    $v->{assign_stack}->push(&merged_array(@as_exprs));
                } # 3項演算子が残っている
            }
        },

        {
            p => "op(lvalue)", # 代入以外の演算子で左辺値で使われるもの
            a => sub {
                my $v = shift;
                # 被演算子を取り出し、それをマージしたもの結果とする
                my $num = int(@{$v->ast_node->{operand}});

                #            print $v->label, " ($num)\n";
                if ($num == 1) { # 左辺の *_ のとき
                    my $as_expr = $v->{assign_stack}->pop;
                    # 近似処理
                    my $has_pointer = 0;
                    foreach (@$as_expr) { # ポインタ変数がいれば 'lr' と近似
                        if ($_->{node}->{stype} =~ /[\*\[]$/) {
                            $_->{ctx} = 'lr';
                            $has_pointer = 1;
                        }
                    }
                    unless ($has_pointer) { # ポインタ変数がないなら、すべて 'lr' と近似
                        map($_->{ctx} = 'lr', @$as_expr);
                    }
                    $v->{assign_stack}->push($as_expr);
                    # 結局、何もしなくてよい?
                } elsif ($num == 2) { # 左辺が _->_, _._, _[_] のとき
                    my $right_expr = $v->{assign_stack}->pop;
                    my $left_expr = $v->{assign_stack}->pop;

                    map($_->{ctx} = 'lr', @$left_expr);
                    $v->{assign_stack}->push(&merged_array($right_expr, $left_expr));
                } # 3項演算子が残っている
            }
        },

        {
            p => "call", # 関数呼出しの場合
            a => sub {
                my $v = shift;

                my $call_node = $v->ast_node;
                my @ast_args = $call_node->children("arg");
                my @args = map($v->{assign_stack}->pop, @ast_args);

                my $fname;
                if ($call_node->child('call')->{t} eq "ID_VF") {
                    $fname = $call_node->child('call')->{name};
                }
                # 関数名が関数呼出しに関する依存関係の定義に含まれていた場合
                if ($fname && exists $fcall->{$fname}) {
                    my @arg_def = @{$fcall->{$fname}};
                    my (@left, @right);
                    @args = reverse(@args); # 引数を出現順に並べ直す
                    for my $i (0..$#args) {
                        my ($is_left, $is_right);
                        if ($i > $#arg_def) { # 指定がなければ右辺値
                            $is_right = 1;
                        } else { # 指定通り
                            $is_right = ($arg_def[$i] =~ /r/);
                            $is_left = ($arg_def[$i] =~ /l/);
                        }
                        my $var;
                        if ($is_left) { # 代入対象となる変数参照を見つける
                            # 型までは確認していないので、特殊な書き方には対応していない
                            $var = &get_left_idvf($ast_args[$i]);
                            if ($var) {
                                push(@left, { node => $var, ctx => 'l' });
                            }
                        }

                        if ($var && !$is_right) { # 単純な代入なら右辺から除く
                            $args[$i] =  [ grep($_->{node} != $var, @{$args[$i]}) ];
                        }
                        push(@right, $args[$i]); # 右辺に相当する変数
                    }

                    push(@right, $v->{assign_stack}->pop); # 関数名に関する式リストを追加
                    my $expr = $call_node->id; # この式のid
                    my $m_right = &merged_array(@right);
                    &record_dependencies($v, $m_right, $expr);

                    foreach (@left) {
                        $v->{var_assign}->{ $_->{node}->uniq_varname } = [ $expr ];
                    }

                    $v->{assign_stack}->push( $m_right ); # 代入式の計算結果
                    $v->cur->{assigned} = 1; ## 式文の終りでの二重登録の回避。ただし、不完全。
                    # ここで代入を登録すると、式文の終りでの解析で、参照する変数を代入している
                    # ノードを登録すると、この関数呼出し自身が含まれる。
                } else {

                    # 引数個文の式リストのリスト
                    push(@args, $v->{assign_stack}->pop); # 関数名に関する式リストを追加

                    # マージしたリストを計算結果とする
                    $v->{assign_stack}->push(&merged_array(@args));
                }
            }
        },

        {
            p => "assign", # 代入の場合
            a => sub {
                my $v = shift;
                my $r_expr = $v->{assign_stack}->pop;  # 右辺の変数参照のリスト
                my $l_var = $v->{assign_stack}->pop;   # 左辺の変数名リスト

                my @left; # 本当の左辺の変数
                my @right; # 左辺で右辺値として参照される変数
                foreach (@$l_var) {
                    if ($_->{ctx} =~ /l/) { # 左辺値の文脈
                        push(@left, $_);
                        push(@right, $_) if $_->{ctx} =~ /r/; # 左辺値と右辺値の両方
                    } else {
                        push(@right, $_);  # 右辺値の文脈
                    }
                }

                my $expr = $v->ast_node->id; # この式のid
                &record_dependencies($v, [ @$r_expr, @right ], $expr);

                foreach (@left) {
                    $v->{var_assign}->{ $_->{node}->uniq_varname } = [ $expr ];
                }

                $v->{assign_stack}->push( [ @left ] ); # 代入式の計算結果
            }
        },

        # # 論理演算子の式を分解したい。しかし、難しい...
        # {
        #     p => qr/^logical_op/,
        #     a => sub {
        #         my $v = shift;
        #         my $expr;
        #         if ($v->{last_cur}->label =~ /^#/) {
        #             $expr = $v->{last_cur}->prev_one->ast_node;
        #         } else {
        #             $expr = $v->{last_cur}->ast_node;
        #         }

        #         # $Data::Dumper::Maxdepth = 4;
        #         # print STDERR "BEFORE\n", Dumper($v->{assign_stack});
        #         my $ref = $v->{assign_stack}->pop;
        #         unless ($v->{last_cur}->label =~ /^#/) {
        #             $v->{assign_stack}->pop;
        #         }

        #         # $Data::Dumper::Maxdepth = 3;
        #         # print STDERR "DEBUG:\n",
        #         #     Dumper($v->{last_cur}->label, $ref, $expr);
        #         &record_dependencies($v, $ref, $expr->id);
        #         my $n = { node => $expr, ctx => $expr->left_ctx,
        #             as_expr => 1 };
        #         # &record_dependencies($v, [ $n ] , $v->ast_node->id);
        #         # print STDERR "DONE\n";
        #         $v->{assign_stack}->push( [ $n ] );
        #         # $Data::Dumper::Maxdepth = 4;
        #         # print STDERR "AFTER\n", Dumper($v->{assign_stack});
        #
        #         # 論理演算子で合流するので、ここで suspend させる必要がある。
        #         # 合流した結果、stack を merge する必要がある。
        #         # stack の merge はコストがかかる? stack の top だけ
        #         # merge させればよい?
        #         #                $v->suspend;
        #     }
        # },

        {
            p => [ qr/^#(if|switch)-end/, # if, switch 文を抜けるとき
                   # "#loop_end-in", # loop を抜けるときは非対応。うまくいかない。
                   qr/^tri_op/,
                   qr/^logical_op/,
                   qr/^label/ ], # 論理積, 論理和, ラベル文
            a => sub {
                my $v = shift;
                $v->suspend;  # 他の経路が終わるまで待つ
            }
        },

        # switch 文で fallthrough がある場合、次の case で合流が発生する
        {
            p => qr/^#(case_.*|default)-in/,
            a => sub {
                my $v = shift;
                if ($v->cur->prev > 1) {
                    $v->suspend;  # 他の経路が終わるまで待つ
                }
            }
        },


        # # 論理演算子を分解したい... 論理演算子に到達する経路を区別するために記録
        # {
        #     end => sub {
        #         my $v = shift;
        #         $v->{last_cur} = $v->cur;
        #     }
        # }

        ];

    ##### 解析処理の表はここまで ######


    # 解析処理の表を visitor に設定
    $visitor->build($tbl);

    # 開始ノードに移動し、visitor を実行する
    foreach ($cfg->node_begin) { # 各 CFG の開始ノードについて
        $visitor->clone->move_to($_)->run;   # そこに移動し、走り始める...
    }

    $self->{pdg} = &unhashed_dest($visitor->{pdg});
    $self->{cfg}->{ast}->{pdg} = $self->{pdg};

    if ($opts->{c}) { # 冗長な制御依存の削除 (時間がかかる)
        print STDERR "the option '-c' may produce an incorrect PDG.\n";
        $self->{ast} = $self->{cfg}->{ast};
        $self->create_graph->remove_redundant_ctrldep;
    }

    return $self;
}

# resume できる条件
sub resumable {
    my @v = @_;  # 待機中の visitors

    my $node = $v[0]->{cur};

    if ($v[0]->label eq "#loop_end-in") {
        do { # まず #cond-end へ移動
            $node = $node->prev_one; # この間、分岐はない。
        } until ($node->label eq "#cond-end");
        $node = $node->pair_begin_one; # #cond-end から #cond-begin へ
        # print STDERR "DEBUG: reached $node->{label}.\n";
    }
    # print STDERR "Resumable: ", int(@{$node->{prev}}), "==", int(@v), "\n";
    # print STDERR "  prev = {", join(", ", map($_->label.":".$_->id, @{$node->{prev}})), "}\n";
    # print STDERR "   node = ", $node->label, ": ", $node->id, "\n";

    # 経路の数と待機している visitor が数が一致していれば、すべて到達
    return int(@{$node->{prev}}) == int(@v);
}

# resume 処理: 変数の表を1つにまとめる。
sub resume {
    my $self = shift;
    my $self_tbl = $self->{var_assign};
    # print STDERR "resume: ", $self->id, "\n";
    foreach my $v (@_) {
        foreach my $k (keys %$self_tbl) {
            $self_tbl->{$k} = &merge_node($self_tbl->{$k}, $v->{var_assign}->{$k});
        }
        $v->move_to_final_state;  # 不要な visitor は終了
        # print STDERR "visitor killed: ", $v->id, "\n";
    }
    $self->{is_suspending} = 0;
}

sub merge_node {
    my ($a, $b) = @_;
    my %node;
    $node{$_} = $_ foreach (@$a);
    $node{$_} = $_ foreach (@$b);
    return [ values %node ];
}


# 式に関する依存関係を記録する関数
sub record_dependencies_for_expr {
    my $v = shift;
    my $as_expr = $v->{assign_stack}->pop;  # 式に対する代入式リスト
    my $expr_node = $v->cur->prev_one; # 直前のノードは式文やreturn文の式に対応

    unless ($expr_node->label =~ /^assign:/ || $expr_node->{assigned}) { # 直前が代入式ではない場合
        # 代入式だった場合は、その時点で依存関係が登録されているので、ここでは登録する必要がない
        &record_dependencies($v, $as_expr, $expr_node->ast_node->id);
    }
}

# return に関する依存関係を記録する関数
sub record_dependencies_for_return {
    my $v = shift;
    my $as_expr = $v->{assign_stack}->pop;  # 返却値に対する代入式リスト

    my $expr_node = $v->cur->prev_one; # 直前の return 文
    &record_dependencies($v, $as_expr, $expr_node->ast_node->id);

    # return 文から関数の出力へのデータ依存
    # 関数の id で記録するので、関数の開始と終了は同じ id になる。
    $v->{pdg}->{data}->{$expr_node->ast_node->id}->{$v->{func}} = 1;
}


# 依存関係を記録する関数
sub record_dependencies {
    my ($v, $as_expr, $expr) = @_;

    my @dep_exprs;
    foreach (@$as_expr) {
        # if (exists $_->{as_expr}) { # 式がノードになるケースへの対応(論理式用)
        #     push(@dep_exprs, $_->{node}->id);
        # } else {
            my $name = $_->{node}->uniq_varname;
            push(@dep_exprs, @{ $v->{var_assign}->{$name} || [] });
        # }
    }
    @dep_exprs = &uniq_id_list(@dep_exprs);

    # 影響を与える代入と代入式の間のデータ依存関係を登録
    $v->{pdg}->{data}->{$_}->{$expr} = 1 foreach (@dep_exprs);

    if (@{$v->{cond}}) { # 制御依存関係を登録
        my $cond = $v->{cond}->top;
        $v->{pdg}->{ctrl}->{$cond}->{$expr} = 1;
    }
    # この式を含む文を記録
    my $stmt = $v->{stmt}->top;
    my $stmt_node = $v->{cfg}->{ast}->node($stmt);
    if ($stmt_node->{t} eq "ST_FOR") { # for 文の場合は初期化式と前進式を文から外す
        my @children = $stmt_node->children('cond');
        if ($expr == $children[1]->id) { # 条件式
            $v->{pdg}->{e2s}->{$expr} = $stmt;
        } else {
            $v->{pdg}->{e2s}->{$expr} = $expr;
        }
    } else {
        $v->{pdg}->{e2s}->{$expr} = $stmt;
    }
    unless ($v->{pdg}->{e2s}->{$expr}) { # CFG と合わない場合がある。そのときはCFGの方修正。
        die "No statement is recorded in stack for:" . $v->{cfg}->{ast}->node($expr)->str;
    }
}

sub get_left_idvf { # 最も左にある変数参照を取り出す
    my $elem = shift;

    my $sym = $elem->op_symbol;
    if ($sym && $sym =~ /^\(_\)|_([_]|->_|._)|[&*]_/) {
        $elem = &get_left_idvf($elem->first_child);
    }
    if ($elem && $elem->{t} eq "ID_VF") {
        return $elem;
    } else {
        return undef;
    }
}

=pod

=head3 conv_stmt_pdg: PDG のノードを文に変換

PDG は標準では式をノードするが、この変換により文を単位とする PDG に変換される。

=cut

sub conv_stmt_pdg {
    my $self = shift;
    my $pdg = $self->{pdg};
    $self->{pdg} = my $pdg_stmt = {'data' => {}, 'ctrl' => {}};

    foreach my $dep ('data', 'ctrl') {  # データ依存関係, 制御依存関係を文単位に変換
        while (my ($from, $to) = each %{$pdg->{$dep}}) {
            my $s_from = $pdg->{e2s}->{$from} || $from;
            $pdg_stmt->{$dep}->{$s_from} = [
                &uniq_id_list( map($pdg->{e2s}->{$_}||$_, @$to) ) ];
        }
    }
    $self->{cfg}->{ast}->{pdg} = $self->{pdg};
    return $self;
}

=pod

=head3 remove_redundant_ctrldep: 冗長な制御依存関係を削除する。

制御依存関係は、条件式に影響を受ける式のうち、その条件式を持つ制御文の
直下に存在する式に対して生成される。しかし、データ依存関係により、
制御依存関係を明示しなくても推移的に制御下にあることがわかる式があり、
それに対して制御依存関係を表現することは冗長であることから削除する。
なお、プログラム依存関係全体を解析するので、時間がかかる。

解析にあたって、条件式に到達したら ctrl 条件式のノードを追加する。
そのあと、制御依存関係の各リンクに分かれて解析が進む。理想的には、
制御文から抜けるときに、ctrl から条件式は抜くべきだが、どの時点で
抜けるかは、現状の PDG ではわからない。しかし、条件文から抜けた
あとに、同じ条件式からの制御依存を受けるノードに遭遇することは
ないので、そのまま維持しても問題がない。

=cut

sub remove_redundant_ctrldep {
    my $self = shift;

    my $visitor = PDG::VISITOR->new;

    # 閉ループ対策。同じ制御の情報を持っている場合に限定する。
    # 解析の順序の影響を受ける可能性あり。
    # $visitor->{env_sig} = sub {
    #     my $v = shift;
    #     return join(":", map($_->id, values %{$v->{ctrl}}));
    #     # 結合する前にid を sort するとより良い気がするが、うまくいかない。
    # };

    # 制御依存を受けるノードを記憶するハッシュ
    $visitor->{ctrl} = {};
    # 各ビジターは訪問記録も含め、独立に動く。1つのノードに複数の経路で
    # 到達する可能性があるため。
#    $visitor->set_deep_copy_keys('ctrl', 'visited');
    $visitor->set_deep_copy_keys('ctrl');
    # visited を複製すると大量の visitor が発生して、まともに動かない。
    # 各ノードに最初に到達する visitor は、必要な条件式ノードは
    # すべて持っている。その後に到達する visitor と差が生じることは
    # あるが、それは、そこまでの経路内で閉じた条件文(繰り返し文を含む)の
    # ものであり、単に余分に持っているだけである。

    $visitor->{analyzer} = sub {
        my $v = shift;

        # 制御依存を与えてくる相手を記録
        # すでに記憶済みなら関係を削除 (unnconnect)
        my $node = $v->cur;

        #################################################
        # 先行するものに対して、ctrl が包含されているなら
        # それ以上、調べなくてよい。
        $node->{checked} ||= {};
        my $checked = $node->{checked};
        my $keep_alive = (%{$v->{ctrl}} ? 0 : 1);
        # print STDERR "keep_alive = $keep_alive\n";
        foreach (values %{$v->{ctrl}}) {
            unless (exists $checked->{$_->id}) {
                $checked->{$_->id} = 1;
                $keep_alive = 1;
                # print STDERR "keep alive\n";
            }
        }

        unless ($keep_alive) {
            $v->move_to_final_state;
            # print STDERR "go final\n";
            return;
        }
        #	print STDERR "keep alive ", $v->id, "\n";
        #################################################

        my @ctrl = $node->prev('ctrl');
        foreach (@ctrl) {
            if (exists $v->{ctrl}->{$_}) {
                $_->unconnect($node, 'ctrl');
            } else {
                $v->{ctrl}->{$_} = $_;
            }
        }
    };

    #$visitor->{final} = sub { my $v = shift; print "##END##\n"};

    # 開始ノードから解析を実行。
    foreach ($self->node_begin) {
        my $v = $visitor->clone;
        $v->move_to($_)->run;
    }

    my $ctrl = {};
    foreach my $node ($self->all_node) {
        $ctrl->{$node->id} = [ map($_->id, $node->next('ctrl')) ];
    }
    $self->{ast}->{pdg}->{ctrl} = $ctrl;

}


=pod

=head3 pdg2dot: PDG を dot 形式に変換

データ依存は実線で、制御依存は点線で表現する。
制御依存のソースノードとなる条件式はダイヤモンド型で、それ以外のノードは楕円で表現される。

=cut

sub pdg2dot {
    my $self = shift;

    $self->create_graph unless ($self->has_graph);

    # GraphViz によるグラフの生成
    my $g = GraphViz->new();  # GraphViz のオブジェクトの生成

    foreach my $node ($self->all_node) {
        foreach my $dst ($node->next('data')) {
            $g->add_edge($node->id, $dst->id);
        }


        if ($node->ast_node->{t} eq "FUNC") {         # 関数の場合: PDG の入口または出口
            if ($node->id =~ /#ret$/) { # 返却値用のダミーノード
                $g->add_node($node->id, label => $node->ast_node->child("name")->attr("name") . ":return",
                shape => "box");
            } else {
                $g->add_node($node->id, label => $node->ast_node->child("name")->attr("name"),
                shape => "box");
                foreach my $dst ( $node->next('ctrl') ) {
                    $g->add_edge($node->id, $dst->id, style=>"dotted", color=>"#A0A0A0");
                }
            }

            next;
        }

        if (my @ctrl = $node->next('ctrl')) { # 制御依存のソースのとき
            foreach my $dst (@ctrl) {
                $g->add_edge($node->id, $dst->id, style=>"dashed");
            }

            my $ast_node = $node->ast_node;
            my $st = $ast_node->{t};
            if ($st =~ /^ST_FOR/) { # 制御文のはず
                $ast_node = ($ast_node->children('cond'))[1]; # 条件式の文字列化
                $st =~ s/^ST_//;
            } elsif ($st =~ /^ST_/) { # 制御文のはず
                $ast_node = $ast_node->child('cond'); # 条件式の文字列化
                $st =~ s/^ST_//;
            } else {
                $st = "";
            }
            if ($st) {
                $st .= "( " . $ast_node->str . " )";
            } else {
                $st = $ast_node->str;
            }
            my $label = $node->id . ": " . &gv_esc($st);
            $g->add_node($node->id, label => $label, shape => "diamond");
        } else {
            my $t = $node->ast_node->str;
            $t = ' ' . $t unless $t; # リテラルの 0 が消えるのを防止
            my $label = $node->id . ": " . &gv_esc($t);
            $g->add_node($node->id, label => $label);
        }

    }
    # PDG を dot 形式で出力
#    return $g->as_text;
    my @out;
    foreach (split("\n", $g->as_text)) {
        # GraphViz のエスケープ問題への対応 (cfg2dot);
        s/\\([<>])/$1/g;
        s/\\/\\\\/g;
        s/&quot;/\\"/g;
        s/&amp;/&/g;
        push(@out, $_);
    }
    return join("\n", @out), "\n";
}


# GraphViz でラベル出力するときのエスケープ処理
# ダブルクォートはエスケープする必要があるが、そのまま表示される
sub gv_esc {
    my $t = shift;
#    print STDERR "DEBUG: $t\n";
    return &CFG::escape_for_GraphViz_bug($t);
#    $t =~ s/\\/\\\\/g; # バックスラッシュ
#    $t =~ s/"/\\"/g; # ダブルクォートをエスケープ
#    return $t;
}

sub merged_array {
    my %all;
    foreach(@_) {
        $all{$_} = $_ foreach (@$_);
    }
    return [ values %all ];
}

sub uniq_id_list {
    my %all;
    foreach(@_) {
        $all{$_} = $_ if $_;
    }
    return values %all;
}

sub unhashed_dest { # PDG の to 側(destination)をハッシュではなくリストする
    my $pdg = shift;

    my $spdg = { 'data' => {}, 'ctrl' => {}, 'e2s' => $pdg->{e2s},
                     func => $pdg->{func}};

    foreach my $edge ('data', 'ctrl') {
        foreach my $node (keys %{$pdg->{$edge}}) {
            $spdg->{$edge}->{$node} = [ keys %{$pdg->{$edge}->{$node}} ];
        }
    }

    return $spdg;
}

=pod

=head3 to_json: PDG を含んだ AST を JSON 化して出力

=cut


sub to_json {
    my $self = shift;
    return $self->{cfg}->to_json;
}


=pod

=head3 load(ast): JSON 化した PDG(PDG を含んだAST)から PDG を再構成

=cut
sub load {
    my $self = shift;
    $self->{ast} = shift;

    unless (exists $self->{ast}->{pdg}) {
        die "Error: AST does not contain a PDG.\n"
    }

    $self->{pdg} = $self->{ast}->{pdg};

    return $self;
}


sub create_graph {
    my $self = shift;

    unless (exists $self->{ast}) {
        die "Error: PDG object did not load a AST.\n"
    }
    my $pdg = $self->{pdg};
    $self->{ast}->generate_node_map;

    foreach my $type ('ctrl', 'data') {
        foreach my $src (keys %{$pdg->{$type}}) {
            my $src_node = $self->create_node($src);
            my $dst_list = $pdg->{$type}->{$src};
            foreach my $dst (@$dst_list) {
                my $dst_node = $self->create_node($dst);
                if ($dst_node->ast_node->{t} eq "FUNC") {
                    # 関数が dst なら、return 文による脱出
                    # 返却値用のダミーノードを作る。
		    my $ret_node = $self->create_node($dst . "#ret");
		    $ret_node->{ast_node} =  $dst_node->{ast_node};
		    $dst_node = $ret_node;
                }
                $src_node->connect($dst_node, $type);
            }
        }
    }

    foreach my $src (keys %{$pdg->{e2s}}) {
        my $src_node = $self->create_node($src);
    }

    return $self;
}

sub has_graph {
    my $self = shift;
    return exists $self->{node_map};
}

sub save_graph {
    my $self = shift;
    $self->{pdg} = {};
    foreach my $node ($self->all_node) {
        foreach my $type ('data', 'ctrl') {
            my @dst = map($_->id, $node->next($type));
            $self->{pdg}->{$type}->{$node->id} = [ @dst ] if @dst;
        }
    }
    $self->{ast}->{pdg} = $self->{pdg};
    return $self;
}


sub create_node {
    my $self = shift;
    my $id = shift;

    if (exists $self->{node_map}->{$id}) {
        return $self->{node_map}->{$id};
    }

    my $node = PDG::NODE->new($self, $id);
    $self->{node_map}->{$id} = $node;
    return $node;
}

sub remove_node {
    my $self = shift;
    my $node = shift;

    foreach my $type ('data', 'ctrl') {
        foreach my $n ($node->next($type)) {
            $n->{prev} = [ grep($_ != $node, $node->prev($type)) ];
        }
        foreach my $n ($node->prev($type)) {
            $n->{next} = [ grep($_ != $node, $node->next($type)) ];
        }
    }
    delete $node->{node_map}->{$node->id};

    return $self;
}

sub all_node {
    my $self = shift;
    return values %{$self->{node_map}};
}

=pod

=head3 node_begin: 関数の PDG の先頭ノードリスト

=cut

sub node_begin {
    my $self = shift;
    return map($self->{node_map}->{$_}, @{$self->{pdg}->{func}});
}

sub search_begin_node {
    my $self = shift;
    my @res;

    foreach my $node ($self->all_node) {
        push(@res, $node) unless $node->prev
    }
    return @res;
}

sub search_end_node {
    my $self = shift;
    my @res;

    foreach my $node ($self->all_node) {
        push(@res, $node) unless $node->next;
    }
    return @res;
}




1;


###########################################################################
package PDG::NODE;

=pod

=head1 パッケージ PDG::NODE のメソッド定義

JSON 形式の PDG を読み込んで(load して)構成された PDG のノードを定義

=head2 PDG::NODE のメソッドの定義

=cut

=pod

=head3 new(pdg, id) : ノードの生成

PDG のノードは、(1)対応する AST のノード、(2)データ依存関係、(3)制御依存関係の
情報を持つ。このうち、(2) と (3) の依存関係は双方向(next, prev)の関係を持つ。

=cut


sub new {
    my $self = bless {};

    my $class = shift;
    $self->{pdg} = shift;
    $self->{id} = shift;

    $self->{ast_node} = $self->{pdg}->{ast}->node($self->{id});
    $self->{data} = { 'next' => [], 'prev' => [] };
    $self->{ctrl} = { 'next' => [], 'prev' => [] };

    return $self;
}

=pod

=head3 connect(src, dst) : ノードの接続

=cut
sub connect {
    my $src = shift;
    my $dst = shift;
    my $type = shift;
    push(@{$src->{$type}->{next}}, $dst);
    push(@{$dst->{$type}->{prev}}, $src);
    return $src;
}

=pod

=head3 connect(src, dst) : ノードの接続

=cut
sub unconnect {
    my $src = shift;
    my $dst = shift;
    my $type = shift;

    $src->{$type}->{next} = [ grep($_ != $dst, @{$src->{$type}->{next}}) ];
    $dst->{$type}->{prev} = [ grep($_ != $src, @{$src->{$type}->{prev}}) ];

    return $src;
}



=pod

=head3 id : ノードの id を返す

=cut
sub id {
    my $self = shift;
    return $self->{id};
}

=pod

=head3 next : 次のノードのリストを返す

引数に 'data' または 'ctrl' を指定する。

=cut

sub next {
    my $self = shift;
    my $type = shift;
    return @{$self->{$type}->{next} || []};
}

=pod

=head3 prev : 前のノードのリストを返す

=cut
sub prev {
    my $self = shift;
    my $type = shift;
    return @{$self->{$type}->{prev} || []};
}

=pod

=head3 ast_node : 対応する AST のノードを返す

引数に 'data' または 'ctrl' を指定する。

=cut

sub ast_node {
    my $self = shift;
    return $self->{ast_node};
}


package PDG::VISITOR;

# ToDo:
#  - PDG::VISITOR の作成
#  - PDG を作るときに、関数のエントリノードと終了ノードを作り
#    制御依存関係を入れる。これにより、開始と終了ノードが一意に定まる(はず)。

use Clone;
use Data::Dumper; #for debug

my $_visitor_id;

# visitor の生成
sub new {
    my $self = bless {};  # ハッシュとして生成
    $self->{pdg} = $_[1]; # visitor が走査する CFG
    $self->{queue} = [];  # 動作待ちの visitor のキュー
    $self->{visit} = {};  # 訪問先を記録するマップ

    # visitor を区別するための id
    $self->{id} = [ ++$_visitor_id ];

    return $self;
}

# visitor の id の文字列表現
sub id {
    my $self = shift;
    return join(".", @{$self->{id}});
}

sub add_id {
    my $self = shift;
    my $id = shift;
    push(@{$self->{id}}, $id);
    return $self;
}

# visitor の移動向きの設定 (next or prev)
sub set_direction {
    my $self = shift;
    $self->{direction} = shift;
    die "illegal direction $self->{direction}."
        unless $self->{direction} =~ /^(next|prev)$/;
    return $self;
}

# visitor の移動向きを next にする
sub set_direction_next {
    my $self = shift;
    return $self->set_direction('next');;
}

# visitor の移動向きを prev にする
sub set_direction_prev {
    my $self = shift;
    return $self->set_direction('prev');;
}

# visitor が現在参照しているノードを返す
sub cur {
    return $_[0]->{cur};
}

# visitor が ctrl と data のどちらから進んできたか。
sub edge {
    my $v = shift;
    return exists $v->{edge} ? $v->{edge} : undef;
}

# visitor を引数のノードに移動させる
sub move_to {
    my $self = shift;
    $self->{cur} = shift;  # current (今、いるノード)
    my $edge = shift;
    $self->{edge} = $edge if $edge;
    return $self;
}

# visitor を1歩進める
sub step {
    my $self = shift;
    my $cur = $self->{cur};


    my @next_data = $cur->next('data');
    my @next_ctrl = $cur->next('ctrl');
    # すでに訪問済みのノードは除外
    @next_data = grep(!$self->already_visited($_), @next_data);
    @next_ctrl = grep(!$self->already_visited($_), @next_ctrl);

    # 訪問すべきノードがないなら、visitor を最終状態にする
    unless (@next_data + @next_ctrl) {
        $self->move_to_final_state;
        return -1;
    }

    my $i = 1;
    if (@next_data) {
        my $n = shift @next_data;
        $self->move_to($n, 'data'); # 次のノードへ進む
        # 訪問すべきノードが残っているなら、クローンを作って、それらを移動させる
        # move_to の副作用を回避するために、先にクローンを作成して移動
        if (@next_data) {
            $self->add_id($i++); # 分岐ごとに異なる id を追加
            map($self->clone->add_id($i++)->move_to($_, 'data'), @next_data);
            # 補足: クローンは queue の中で待機
        }
    } else {
        my $n = shift @next_ctrl;
        $self->move_to($n, 'ctrl'); # 次のノードへ進む
    }
    if (@next_ctrl) {
        $self->add_id($i++); # 分岐ごとに異なる id を追加
        map($self->clone->add_id($i++)->move_to($_, 'ctrl'), @next_ctrl);
        # 補足: クローンは queue の中で待機
    }

    return 0;
}

# visitor の clone を生成する
sub clone {
    my $self = shift;
    my $clone = bless { %$self }; # 属性値のコピーを持つクローンを生成
    # ここでの copy は浅い(shallow)ので、visitor に属性を追加している場合、
    # その属性値を正しくコピーしているとは限らない。
    # 追加した属性に関するコピー方法は、必要があれば、clone_copy に無名ルーチンとして
    # 定義しておく。

    # 指定されたキーについて deep copy を行う。(loop_stack を含む)
    foreach (@{$self->{deep_copy_keys}||[]}) {
        $clone->{$_} = Clone::clone($self->{$_});
    }

    if (defined $self->{clone_copy}) {  # その定義があれば実行する。
        &{$self->{clone_copy}}($self, $clone);
    }

    # クローンはすぐには動かないので、queue に入れておく。
    push(@{$self->{queue}}, $clone);
    return $clone;  # 生成したクローンを返す
}

sub set_deep_copy_keys {
    my $self = shift;
    $self->{deep_copy_keys} = [ @_ ];
    return $self;
}

sub add_deep_copy_keys {
    my $self = shift;
    push(@{$self->{deep_copy_keys}}, @_);
    return $self;
}


# visitor を動かす
sub run {
    my $self = shift;
    while ($self) {  # visitor が存在する限り、動き続ける
        if ($self->is_final || $self->is_suspending) { # visitor が最終状態または待機なら、
            $self = $self->unqueue; # queue のクローンと交代する
            next;  # 交代すべきクローンがない場合があるので、while の条件式に戻る
        }
        $self->analyze;    # visitor が解析処理をする
        $self->mark_visited; # 現在のノードの訪問記録を付ける
        last if $self->is_suspending;
        unless ($self->is_final) {
            $self->step;         # visitor を1歩進める
        }
    }
}

# visitor が最終状態かどうを判定する
sub is_final {
    return $_[0]->{is_final_state};  # マークの有無で判定
}

sub continue {
    $_[0]->{is_final_state} = 0;
    $_[0]->run;
}

# visitor を最終状態に移行する
sub move_to_final_state {
    my $self = shift;
    $self->{is_final_state} = 1;  # 最終状態と記録する
    delete $self->{is_suspending};  # suspend の状態からは削除

    # 最終状態になるときに後始末が必要なら実行する
    if (defined $self->{final}) {
        return &{$self->{final}}($self);
    }
}

sub is_suspending {
    return $_[0]->{is_suspending};
}

sub suspend {
    my $self = shift;
    $self->{cur}->{suspend_queue} ||= [];
    my $sq = $self->{cur}->{suspend_queue};

    #    print STDERR "SUSPEND called!\n";
    #もし待ち状態を解消する条件が揃っていたら resume する。
    if ($self->resumable(@$sq)) {
        #	print STDERR "Resumed.\n";
        $self->resume(@$sq);
        $self->{cur}->{suspend_queue} = [];
    } else {
        #	print STDERR "Suspended.\n";
        $self->{is_suspending} = 1;
        push(@{$self->{queue}}, $self);  # 動作待ちキューに追加
        push(@$sq, $self); # そのノードの待ちキューに追加
    }
    #    print STDERR "  label: ", $self->label, "\n";
}

sub resumable {
    my $self = shift;
    unless (exists $self->{resumable}) {
        die "No resumable checking method exists.";
    }
    return $self->{resumable}->($self, @_);
}


sub resume {
    my $self = shift;

    unless (exists $self->{resume}) {
        die "No resume method exists.";
    }
    return $self->{resume}->($self, @_);

    map($_->{is_suspending} = 0, ($self, @_));

    # 次の step で余分に進まないようにするためのフラグ
    # $self->{just_after_resumed};
}

sub unqueue {
    my $self = shift;
    my $q = $self->{queue};
    return undef unless @$q; # キューが空の場合
    #    print STDERR "Length of QUEUE: ", int(@$q), "\n";
    my $i;
    for ($i = 0; $i < @$q; $i++) {
        unless ($q->[$i]->is_suspending) {
            last;
        }
        #	print STDERR "$i in QUEUE is suspending.\n";
    }
    if ($i == @$q) {
        die "Dead lock error: All visitors suspend for waiting others.";
    }
    my $n = splice(@$q, $i, 1);
    return $n;
}


# queue で待機中の visitor を破棄する
sub kill_queue {
    my $self = shift;
    $self->{queue} = [ ];
}

# visitor が分析処理を行う。処理内容は analyzer 属性に定義
sub analyze {
    my $self = shift;
    &{$self->{analyzer}}($self) if $self->{analyzer};
}

# visitor が現在のノードを訪問済みかどうかを判定
sub already_visited {
    my $self = shift;
    my $node = shift;

    # signature が定義されていなければ常に偽を返す
    return 0 unless defined $self->{env_sig};

    # 判定は、signature が一致するかどうかで行う
    return (defined $self->{visited}->{$node}
            && $self->{visited}->{$node} eq $self->env_sig);
    # 注意: すでに訪問していても、signature が一致しなければ
    # 訪問していないという扱いになる。
}

# visitor が現在のノードを訪問済みと記録
sub mark_visited {
    my $self = shift;

    $self->{visited}->{$self->{cur}} = $self->env_sig;
}

# visitor が訪問時に使用する signature を返す (environment signature)
# 具体的な手続きは属性 env_sig に無名ルーチンとして登録する。
sub env_sig {
    my $self = shift;
    if (defined $self->{env_sig}) {
        return &{$self->{env_sig}}($self);
    } else {
        return undef;
    }
}

# ASTのノードの wrapper (AST::NODE) を返す
sub ast_node {
    return $_[0]->{cur}->ast_node;
}

# visitor が訪問中のCFGノードのラベルを返す
sub label {
    my $self = shift;
    return $self->{cur}->label;
}

# visitor が訪問中のCFGノードのラベルの型名のみを返す
sub label_type {
    my $self = shift;
    my ($type) = ($self->{cur}->label =~ /^([^:]*)/);

    return $type;
}

# visitor が訪問中のCFGノードの id を返す
sub node_id {
    my $self = shift;
    return $self->{cur}->id;
}

# visitor の解析器を表から構築
sub build {
    my $self = shift;
    my $tbl = shift;
    # ラベルのパターンとそれに対する操作(無名サブルーチン)のリスト
    #  [ { p => "assign", a => sub { ... } }, ... ]
    # 無名サブルーチンの引数はビジターとラベル
    # ラベルのパターンは文字列または正規表現(qr/.../)

    my $t = { reg => [], str => {} };
    my $begin;
    foreach my $e (@$tbl) {
        if (exists $e->{begin}) {
            $begin = $e->{begin};
        } elsif (exists $e->{p}) {
            my @p = ref($e->{p}) eq "ARRAY" ? @{$e->{p}} : ($e->{p});

            foreach my $p (@p) {
                if (ref($p) eq "Regexp") {
                    push(@{$t->{reg}}, { p => $p, a => $e->{a} });
                } else {
                    $t->{str}->{$p} = $e->{a}
                }
            }
        }
    }

    $self->{analyzer} = sub {
        my $v = shift; # visitor
        my $str_tbl = $t->{str};
        my $type = $v->label_type;
        my $label = $v->label;

        $begin->($v) if ($begin);

        if (exists $str_tbl->{$type}) {
            $str_tbl->{$type}->($v, $type);
        } elsif (exists $str_tbl->{$label}) {
            $str_tbl->{$label}->($v, $label);
        } else {
            foreach my $e (@{$t->{reg}}) {
                if (my @p = ($label =~ $e->{p})) {
                    $e->{a}->($v, @p);
                }
            }
        }
    };  # end of sub
}


1;

