#!/usr/bin/env perl
# 
# Copyright (c) 2009-2025 The TEBA Project. All rights reserved.
# 
# Redistribution and use in source, with or without modification, are
# permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# Author: Atsuhi Yoshida


use strict;
use warnings;
use GraphViz;

use URI::Escape;

# for debug
#use Data::Dumper;


use AST;

use Getopt::Std;
my %opts = ();

if (!getopts("phi", \%opts) || $opts{h}) {
    print STDERR "ast2dot.pl [-ph] [json-based-ast-file]\n",
                 "  -p: output as PNG format.\n",
                 "  -i: node label with id.\n",
                 "  -h: help.\n";
    exit(1);
}

my $root = AST::NODE->build_from_json_text(join('', <>));

my $g = GraphViz->new();

&tree2dot($g, $root);

if ($opts{p}) {
    print $g->as_png;
} else {
    # print $g->as_text;
    my $t = $g->as_text;
    my @out;
    foreach my $line (split("\n", $t)) {
        # GraphViz のエスケープ問題への対応 (cfg2dot);
        $line =~ s/\\([<>])/$1/g; # 勝手にエスケープされているので、戻す。
        if ($line =~ m/^(\s*label="\w+)_(.*)$/) { # 独自にエスケープした型
            $line = uri_unescape("$1$2");
            $line =~ s/\\/\\\\/g;
            $line =~ s/&quot;/\\"/g;
            $line =~ s/&amp;/&/g;
        }
        # ノードの幅がエスケープ処理に影響するので、再計算するよう削除する。
        $line =~ s/width=\d+\.\d+//;
        push(@out, $line);
    }
    print join("\n", @out), "\n";
}

sub tree2dot()
{
    my ($g, $el) = @_;

    my $label = &createLabel($el);
#    $label =~ s/\"/\\"/g;  # GraphViz.pm has a bug escaping double quotes twice.
    $g->add_node($el->{id}, label => "$label");
    if (&isDirective($el)) {
	$g->add_node($el->{id}, shape => "hexagon", style=>"filled", color=>"lightgray");
    }
    foreach my $ch ($el->children_all) {
	&tree2dot($g, $ch);
	$g->add_edge($el->{id}, $ch->{id});
    }
    return $el;
}

sub createLabel()
{
    my $el = shift;
    my $res = $el->{t};

    if ($el->{t} =~ /^ID_/) {
	$res .= " : " . $el->{name};
    } elsif ($el->{t} =~ /^P/) {
	if ($el->{sym}) {
	    $res = "op: " . $el->{sym};
	} elsif (exists($el->{call})) {
	    $res = "call";
	}
    } if ($el->{t} =~ /^LI/) {
	$res .= " : " . $el->{value};
    }
    $res .= ":". $el->id if $opts{i};

    return &escape_for_GraphViz_bug($res);
}

sub escape_for_GraphViz_bug {
    my $label = shift;
    if ($label =~ /['"]/) { # 文字列や文字リテラル
        # GraphViz.pm が dot に変換するときのエスケープ処理がおかしい。
        # 一度、HTML の表現方式に直して、最後に出力するときに戻すようにする。
        # 型名の後ろに "_" を追加して、エスケープしている印にする
        $label =~ s/^(\w+)\s*:/$1_:/;
        $label =~ s/&/&amp;/g;
        $label =~ s/"/&quot;/g;
        $label = uri_escape($label);
    }
    return $label;
}

sub isDirective {
    my $el = shift;
    return $el->{t} =~ /^DIRE_/;
}

sub isObj() {
    return ref($_[0]) eq "HASH";
}
