#!/usr/bin/env perl
# 
# Copyright (c) 2009-2025 The TEBA Project. All rights reserved.
# 
# Redistribution and use in source, with or without modification, are
# permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# Author: Atsuhi Yoshida


# 良いプログラムを書くためのおまじない。
use warnings;
use strict;

# debug 用
use Data::Dumper;
#use Carp qw(confess);
#$SIG{__DIE__} = \&confess;


# TEBA のライブラリの読み込みパスの設定 (TEBA のパッケージに入っているコマンド用)
# TEBA のパッケージに入っていない場合は PERLLIB 設定する必要がある。
use FindBin qw($Bin);
use lib "$Bin/../TEBA";

# TEBA の CFG, PDG モジュールを使用。
use PDG;

# オプションの設定
use Getopt::Std;

my %opts = ();
if (!getopts("csf:dh", \%opts) || $opts{h}) {
    print STDERR "ast2pdg.pl [-sdh] [-f def-file ] [TEBA AST JSON file]\n",
        "  -c: remove redundant control dependencies.\n",
        "  -s: statement mode.\n",
        "  -f [func call def file]: def analysis of function calls.\n",
        "  -d: debug mode.\n",
        "  -h: help.\n";
    exit(1);
}

# 関数呼出しの代入の定義の読み込み
# 1行に1の呼出しの定義
#  関数名(引数定義)
# 引数定義は、以下のものをカンマで区切って書く
#   l: 代入
#   r: 参照
#   lr: 代入と参照
# 例: sprintf(l, r)
# 定義より実引数が多い場合は r として扱う。
my $fcall = $opts{f} ? &parse_func_arg_ctx($opts{f}) : {};

# JSON 形式の AST を読み込む
my $ast = AST::NODE->build_from_json_text(join('', <>));

# 読み込んだ AST から CFG オブジェクトを生成する
my $cfg = CFG->new($ast);

my $pdg = PDG->new($cfg, $fcall, \%opts);

# 文単位の CFG が必要な場合
if ($opts{s}) {
    $pdg->conv_stmt_pdg;
}

# JSON 形式にして出力
print $pdg->to_json;

sub parse_func_arg_ctx {
    my $fname = shift;

    my $fcall = {};
    open(my $f, '<', $fname) || die "Error: $!: $fname";
    foreach (<$f>) {
        if (my ($fname, $arg) = /^(\w+)\((.*)\)/) {
            $arg =~ s/\s+//g;
            $fcall->{$fname} = [ split(",", $arg) ];
            # 引数をスタックから取り出す順番に合わせて逆順にしておく
        }
    }
    close($f);
    return $fcall;
}
