#!/usr/bin/env perl
# 
# Copyright (c) 2009-2025 The TEBA Project. All rights reserved.
# 
# Redistribution and use in source, with or without modification, are
# permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# Author: Atsuhi Yoshida


# ToDo:
# - malloc の実装
# - 関数ポインタへの対応
# - 未定義変数の参照の扱い
# - union への対応 (たぶん、無理)

# Done:
# - キャスト演算子への対応
# - enum への対応
# - グローバル変数の初期化への対応
# - 3項演算子の動作の確認
# - 多重配列のポインタ演算
#     int ary[5][10] = { 0 }; int (*p)[10] = ary; p++; *p[1] = 99; p++; *p[1] = 999;


use warnings;
use strict;
use v5.10;

use FindBin qw($Bin);
use lib "$Bin/../TEBA";

use Data::Dumper;
use Carp qw(confess);
$SIG{__DIE__} = \&confess;

use CFG;
use CParser;

use Getopt::Std;
my %opts = ();


if (!getopts("cdh", \%opts) || $opts{h}) {
    print STDERR "c_intp.pl [-ch] [files...]\n",
        "  -c: parse input as C source code.\n",
        "  -d: debug mode.\n",
        "  -h: help.\n";
    exit(1);
}

# ソースコードを読み込み構文解析。JSON データで出力。
my $input_text = join('', <>);
my $json_text = $opts{c} ?
    CParser->new->with_symboltable->as_json->parse($input_text)
    : $input_text;

# AST を読み込む
my $ast = AST::NODE->build_from_json_text($json_text);

# 読み込んだ AST から expression level の CFG オブジェクトを生成する
my $cfg = CFG->new($ast);

# CFG 上を動く visitor を作成。解釈実行するので、順方向に移動する。
my $visitor = $cfg->visitor->set_direction_next;

# visitor に計算のための stack と、変数の表を作成
# visitor 間で共有する。
$visitor->{stack} = CFG::STACK->new;
$visitor->{var} = VARTBL->new($ast);

$visitor->{var}->register_func( C->func_list );

foreach my $n ($cfg->node_begin) {
    my $name_node = $n->ast_node->child('name');
    next unless ($name_node);  # 宣言しかない場合には、実行できない。
    $visitor->{var}->alloc_var($name_node)
        ->assign(VALUE::FUNC->new($n->label));
}


# 各ノードでの処理を行う手続きを登録
$visitor->build(&analyzer_tbl);

#$visitor->{env_sig} = undef;
#$visitor->{clone_copy} = undef;
# visitor は、繰返しでは同じノードを何度も通過するので、訪問記録は不要。
# よって、env_sig は undef のままにする。
# stack と var は、visitor 間で共有するので、shallow copy されればよい。
# 複製のための特別な処理は不要なのえ、clone_copy も undef のままにする。


# 開始ノードに移動し、実行
## 関数定義外の処理 unit の実行
my $entry = $cfg->node_begin("#unit");
my $exit = $cfg->node_end("#unit");

my $main = $cfg->node_begin("#func-main");
if ($main) {
    $exit->connect($main);
}

$visitor->move_to($entry);
$visitor->run;

if ($opts{d}) {
    print "\n## Variable Table ##\n";
    print $visitor->{var}->dump;
    print "\n## Stack Dump ##\n";
    $Data::Dumper::Maxdepth = 5;
    print Dumper($visitor->{stack}); # スタックを出力
}


# 解釈処理の表
sub analyzer_tbl {
    return [
        {   # 各ノードでの解析の先頭で行う共通処理 (主にデバッグ用)
            begin => sub {
                return unless $opts{d};
                my $v = shift;
                print STDERR "----\nVISITOR: id = ", $v->short_id, "\n";
                print STDERR "Node: ", $v->label, ":", $v->cur->id, "  (", $v->label_type, ")\n";
                print STDERR "Var:\n", $v->{var}->dump("  ");
#                print STDERR "Stack ", Dumper($v->{stack});
                print STDERR "Stack ", dump_stack($v->{stack});
                print STDERR "\n";
            }
        },
        {
            p => "#func-main-begin",
            a => sub {
                my $v = shift;
                $v->{stack}->push(VALUE->literal(0, 'int'));
                $v->{stack}->push(VALUE->literal(0, 'int')); # NULL
            }
        },

        {
            p => qr/^#func-.*-begin$/,
            a => sub {
                my $v = shift;
                $v->{var}->push_frame;
            }
        },
        {
            p => qr/^#func-.*-end$/,
            a => sub {
                my $v = shift;
                $v->{var}->pop_frame;
            }
        },
        {
            p => "arg", # 関数の仮引数
            a => sub {
                my $v = shift;
                my $var = $v->{var}->alloc_var($v->ast_node);
                $var->assign($v->{stack}->pop);
            }
        },

        {
            p => "literal", # 定数値
            a => sub {
                my $v = shift;
                my $value;
                if ($v->ast_node->attr('t') eq "LIS") {
                    $value = VALUE->literal_str($v->ast_node->attr('value'),
                                                $v->{var});
                } elsif ($v->ast_node->attr('t') eq "LIC") {
                    my $char = eval($v->ast_node->attr('value'));
                    my $char_code = $char eq "\\0" ? 0 : ord($char);
                    $value = VALUE->literal($char_code, 'char');
                } else {
                    $value = VALUE->literal($v->ast_node->attr('value'),
                                            $v->ast_node->attr('stype'));
                }
                $v->{stack}->push($value);
            }
        },

        {
            p => "d_value", # 宣言された変数
            a => sub {
                my $v = shift;
                my $var = $v->{var}->alloc_var($v->ast_node);
                $v->{stack}->push($var);
            }
        },

        {
            p => "rvalue", # 変数の値の参照(右辺値)
            a => sub {
                my $v = shift;
                my $type = $v->ast_node->attr('stype');
                my $var = $v->{var}->alloc_var($v->ast_node);
                my $res;
                if ($type =~ /\[$/) {
                    $res = VALUE::POINTER->new_from_ary($var); # 配列
                } else {
                    $res = $var->value; # 変数の値を push
                }
                $v->{stack}->push($res);
            }
        },
        {
            p => "lvalue", # 左辺値 (変数の参照)
            a => sub {
                my $v = shift;
                my $var = $v->{var}->alloc_var($v->ast_node);
                $v->{stack}->push($var); # 変数参照を返す
            }
        },
        {
            p => "lrvalue", # increment など、左辺値と右辺値の両方がある場合
            a => sub {
                my $v = shift;
                my $var = $v->{var}->alloc_var($v->ast_node);
                $v->{stack}->push($var, $var->value); # 変数参照と変数の値を push
            }
        },

        {
            p => qr/(l?)(r?)member/, # メンバの参照。参照演算子に情報を渡す。
            a => sub {
                my $v = shift;
                my ($l, $r) = @_; # left and right
                $r = 1 unless ($l || $r); # "member" の場合は値参照(right)
                $v->{stack}->push(
                    VALUE::MEMBER->new($l, $r, $v->ast_node->attr('name')) );
            }
        },

        {
            p => "assign", # 代入の場合
            a => sub {
                my $v = shift;
                my $sym = $v->ast_node->op_symbol;
                my $rvalue = $v->{stack}->pop;  # 代入する値 (右辺)
                my $var = $v->{stack}->pop;     # 変数参照 (左辺)
                if ($sym eq "_=_") {  # 普通の代入
                    $var->assign($rvalue);
                    $v->{stack}->push($var->value); # 左辺の変数の値が全体の評価値
                } elsif ($sym eq "_++") { # 後置 increment の場合
                    $v->{stack}->push($rvalue);  # increment 前の値が全体の評価値
                    $var->assign($rvalue->inc); # 変数の値を1増やす
                } elsif ($sym eq "++_") { # 前置 increment の場合
                    $var->assign($rvalue->inc); # 変数の値を1増やす
                    $v->{stack}->push($var->value); # increment 後の値が全体の評価値
                } elsif ($sym eq "_--") { # 後置 decrement の場合
                    $v->{stack}->push($rvalue);  # decrement 前の値が全体の評価値
                    $var->assign($rvalue->dec); # 変数の値を1減らす
                } elsif ($sym eq "--_") { # 前置 increment の場合
                    $var->assign($rvalue->dec); # 変数の値を1減らす
                    $v->{stack}->push($var->value); # でcrement 後の値が全体の評価
                } elsif ($sym =~ /^_(.)=_$/) { # 複合代入演算子の扱い
                    my $op = "_".$1."_"; # 演算子を準備
                    my $var_val = $var; # 2つ目の pop で得られるのは変数の値
                    $var = $v->{stack}->pop; # 3つ目に変数参照が存在
                    $var->assign( $var_val->op($op, $rvalue) );
                    $v->{stack}->push($var->value);
                } else {
                    die "Not supported yet: ".$v->ast_node->op_symbol.".\n";
                }
            }
        },

        {
            p => qr/^op(?:\((l)(r?))?/, # 代入以外の演算子
            a => sub {
                my $v = shift;
                my @ctx = @_; # l and r
                my $sym = $v->ast_node->op_symbol;
                my $res;
                if ($sym eq "_._") { # メンバの参照
                    my $mem = $v->{stack}->pop; # メンバの情報
                    my $name = $mem->{name};

                    my $val;
                    if ($mem->{r}) {
                        my $st = $v->{stack}->pop;
                        my $m = $st->member($name);
                        $val = ($m->{st} =~ /\[$/) ?
                            VALUE::POINTER->new_from_ary($m) : $m->value;

                    }
                    my $var;
                    if ($mem->{l}) {
                        my $st_var = $v->{stack}->pop;
                        $st_var->alloc_struct;
                        $var = $st_var->value->member($name);
                    }

                    $v->{stack}->push($var) if $var;
                    $v->{stack}->push($val) if $val;
                    return;
                } elsif ($sym eq "_->_") { # メンバの参照
                    my $mem = $v->{stack}->pop; # メンバの情報

                    my $st = $v->{stack}->pop;
                    $st = $st->deref->alloc_struct;
                    my $m = $st->value->member( $mem->{name} );

                    $v->{stack}->push($m) if $mem->{l};
                    if ($mem->{r}) {
                        my $val = ($m->{st} =~ /\[$/) ?
                            VALUE::POINTER->new_from_ary($m) : $m->value;
                        $v->{stack}->push($val);
                    }
                    return;

                } elsif ($sym =~ /^_.+_$/) { # 2項演算子
                    my @val = ($v->{stack}->pop, $v->{stack}->pop);
                    # 演算結果を push
                    $res = $val[1]->op($sym, $val[0]);
                } elsif ($sym eq "*_") {
                    my $ptr = $v->{stack}->pop->clone;

                    # 演算子の適用(加算と間接参照)
                    $ptr->{st} =~ s/\.[\[\*]$//;
                    shift(@{$ptr->{size}});

                    my $is_elem = ($ptr->{st} !~ /[\[\*]$/); # 配列の要素である

                    # ポインタのときはそのまま、要素のときは参照先を取り出す
                    $res = $is_elem ? $ptr->deref : $ptr;

                    if ($ctx[0]) { # 左辺の文脈あり
                        $v->{stack}->push($res) if ($ctx[1]); # 右辺の文脈もあるなら演算結果も push
                    } else {
                        $res = $res->value if $is_elem; # 右辺で要素の参照はその値にする
                    }

                } elsif ($sym eq "&_") {
                    my $var = $v->{stack}->pop;
                    $res =  VALUE::POINTER->new($var);
                } elsif ($sym eq "_[_]") {
                    my ($ind, $ary) = ($v->{stack}->pop, $v->{stack}->pop->clone);

                    if ($ary->{st} =~ '\*$') { # ポインタの参照の場合
                        $ary = VALUE::POINTER->new_from_ary($ary->deref); # 配列化
                    }

                    # 演算子の適用(加算と間接参照)
                    $ary->{st} =~ s/\.[\[\*]$//;
                    $ary->deref->{index} += $ind->{v} * shift(@{$ary->{size}});

                    my $is_elem = ($ary->{st} !~ /[\[\*]$/); # 配列の要素である

                    # ポインタのときはそのまま、要素のときは参照先を取り出す
                    $res = $is_elem ? $ary->deref : $ary;

                    if ($ctx[0]) { # 左辺の文脈あり
                        $v->{stack}->push($res) if ($ctx[1]); # 右辺の文脈もあるなら演算結果も push
                    } else {
                        $res = $res->value if $is_elem; # 右辺で要素の参照はその値にする
                    }
                } elsif ($sym eq "T_") { # cast operator
                    $res = $v->{stack}->pop;
                    $res->{st} = $v->ast_node->attr('stype');
                    if ($res->{st} =~ /\bint$/) {
                        $res->{v} = int($res->{v});
                    }
                } else { # 単項演算子 (3項演算子は含まれない)
                    my $val = $v->{stack}->pop;
                    $res = $val->op($sym);
                }
                $v->{stack}->push($res); # 演算結果を push
            }
        },
        {
            p => "call", # 関数呼出し
            a => sub {
                my $v = shift;
                my $func = $v->{stack}->pop;
                $func->call($v);
            }
        },
        {
            p => qr/d_op(\(lvalue\))?: _\[_\]/, # 宣言子の中の配列指定
            a => sub {
                my $v = shift;
                my $size = $v->{stack}->pop;
                my $ary = $v->{stack}->pop;
                $ary->{buf}->{size} ||= [];
                if ($ary->{st} =~ /\[$/) { # 配列の場合
                    push(@{$ary->{buf}->{size}}, $size->{v});
                    $ary->{buf}->{max_num} = ($ary->{buf}->{max_num} // 1) * $size->{v};
                    $ary->{st} =~ s/\.\[$//;
                } else {
                    push(@{$ary->{size}}, $size->{v});  # ポインタの場合 (int (*p)[]; など)
                    $ary->{max_num} = ($ary->{max_num} // 1) * $size->{v};
                    $ary->{st} =~ s/\.\*$//;
                    $ary->{st} =~ s/\.\[$/.*/;  # 配列ではなくポインタとして扱う
                }
                $v->{stack}->push($ary);
            }
        },
        {
            p => "d_op: _[]", # 宣言子の中の配列指定 (サイズなし)
            a => sub {
                my $v = shift;
                my $ary = $v->{stack}->pop;
                $ary->{buf}->{size} ||= [];
                push(@{$ary->{buf}->{size}}, undef);
                $v->{stack}->push($ary);
            }
        },
        {
            p => "#initlist-begin",
            a => sub {
                my $v = shift;
                $v->{stack}->push(VALUE->mark($v->label));
            }
        },

        {
            p => "#initlist-end",
            a => sub {
                my $v = shift;
                my @vals;
                while (1) {
                    my $x = $v->{stack}->pop;
                    last if ($x->{t} eq "#initlist-begin");
                    unshift(@vals, $x);
                }
                $v->{stack}->push(VALUE->array(@vals));
            }
        },

        {
            p => qr/^#(expr_st|for_(init|succ)|decr)-end$/,
            # スタックに計算結果が残るので、削除
            # 対象: 式文、for 文の初期化式、前進式、初期化を伴う宣言子
            a => sub {
                my $v = shift;
                $v->{stack}->pop;
            }
        },

        {
            p => "#true", # 分岐で真の場合に進んだら
            a => sub {
                my $v = shift;
                my $val = $v->{stack}->top;
                unless ($val->{v}) {  # 偽ならこの visitor は破棄
                    $v->move_to_final_state;
                } else {          # 真ならこの visitor は継続するので、他の visitor を破棄
                    # 次が論理演算子なら、計算結果を維持しておく必要がある。
                    unless ($v->cur->next_one->label =~ /^logical_op/) {
                        $v->{stack}->pop;
                    }
                    $v->kill_queue;
                }
            }
        },

        {
            p => "#false", # 分岐で偽の場合に進んだら
            a => sub {
                my $v = shift;
                my $val = $v->{stack}->top;
                if ($val->{v}) {  # 真ならこの visitor は破棄
                    $v->move_to_final_state;
                } else {          # 偽ならこの visitor は継続するので、他の visitor を破棄
                    # 次が論理演算子なら、計算結果を維持しておく必要がある。
                    unless ($v->cur->next_one->label =~ /^logical_op/) {
                        $v->{stack}->pop;
                    }
                    $v->kill_queue;
                }
            }
        },

        ];
}

sub dump_stack {
    my $stack = shift;
    my @out;
    foreach ($stack->list) {
        my $str = $_->str;
        $str =~ s/\n/\\n/g;
        $str =~ s/\t/\\t/g;
        push(@out, "    " . (ref $_ ? $str : $_));
    }
    return join("\n", "[", @out, "]\n");
}


##########################################################################
package VARTBL;

use Data::Dumper;

sub new {
    my ($class, $ast) = @_;
    my $obj = bless { frame => [], top => 0, symtbl => SYMTBL->new($ast) };
    # 構造体のメンバを扱うために、AST の記号表が必要。

    $obj->{frame}->[ $obj->{top} ] = { var => [] }; # 空のテーブルの準備
    return $obj;
}

sub frame_top {
    my $self = shift;
    return $self->{frame}->[ $self->{top} ];
}

sub frame_base {
    my $self = shift;
    return $self->{frame}->[ 0 ];
}

sub alloc_var_entry {
    my ($self, $name, $type, $scope) = @_;
    my $var = $self->lookup_var($name);
    return $var if $var; # すでに確保済なら、それを返す。

    my $buf = { name => $name, st => $type, scope => $scope, mem => [],
                vartbl => $self };
    push(@{$self->frame_top->{var}}, $buf);
    $self->frame_top->{named_var}->{$name} = $buf if $name;
    my $res = VAR_ENTRY->new($buf);

    if ($type =~ /^enum (\w+)$/) { # 列挙子の場合
        my $tag = $1;
        $buf->{st} = "int";
        my $enum = $self->{symtbl}->{scope}->{$scope}->lookup("tag:$tag");
        my ($mem) = ($name =~ /^(\w+):/);
        if (exists $enum->{$mem}) {
            $res->assign(VALUE->literal($enum->{$mem}, "int"));
        }
    }

    return $res;
}

sub alloc_var {
    my ($self, $ast) = @_;
    my $name = $ast->uniq_varname;
    my $type = $ast->attr('stype');
    my $scope = $ast->attr('scope');

    return $self->alloc_var_entry($name, $type, $scope);
}

sub push_frame {
    my $self = shift;
    $self->{top}++;
    $self->{frame}->[ $self->{top} ] = { var => [] };
    return $self;
}

sub pop_frame {
    my $self = shift;
    $self->{frame}->[ $self->{top} ] = undef;
    $self->{top}--;
    return $self;
}

sub lookup_var {
    my $self = shift;
    my $name = shift;
    return undef unless $name;
    if (exists $self->frame_top->{named_var}->{$name}) {
        return VAR_ENTRY->new( $self->frame_top->{named_var}->{$name} );
    } elsif (exists $self->frame_base->{named_var}->{$name}) {
        return VAR_ENTRY->new( $self->frame_base->{named_var}->{$name} );
    }
    return undef;
}

sub register_func {
    my $self = shift;
    my $func_list = shift;
    foreach (@$func_list) {
        $self->alloc_var_entry("$_->{name}:#GLOBAL", $_->{type})
            ->assign(VALUE::FUNC->new($_->{func}));
    }
}

sub dump {
    my $self = shift;
    my $indent = shift || "";
    my @res;
    my $i = 0;
    foreach my $f (@{$self->{frame}}) {
        last unless $f;
        push(@res, $indent, "____ frame ", $i++, " ____\n");
        foreach (@{$f->{var}}) {
            my @out;
            my $name = $_->{name} || "#NONE";
            if (exists $_->{size}) {
                my $s = join(", ", map(defined $_ ? $_ : "#UNDFF",
                                       @{$_->{size}}));
                $name .= " (size = [ $s ])";
            }
            my $is_ary = ($_->{st} =~ /[\*\[]$/ || @{$_->{mem}} > 1);
            foreach my $val (@{$_->{mem}}) {
                unless (defined $val) {
                    push(@out, "#UNDEF");
                    next;
                }
                my $str = $val->str;
                $str =~ s/\n/\\n/g;
                $str =~ s/\t/\\t/g;
                push(@out, $str);

                if ($val->{t} eq "pointer" && $val->{st} eq 'char.*'
                    || $val->{t} eq "array") {
                    $is_ary = 0;
                }
            }
            @out = (join(",", @out));
            @out = ("[ ", @out, " ]") if $is_ary;
            @out = ($indent, "$name :\t", @out, " :$_->{st} \n");
            push(@res, @out);
        }
    }
    return join("", @res);
}

1;
##########################################################################
package VAR_ENTRY;

use Data::Dumper;

sub new {
    my ($class, $buf) = @_;
    return bless { st => $buf->{st}, buf => $buf, index => 0 };
    # var は symbol table を持つようにするべき? assign で sym_tbl を渡すのはウザイ
}

sub assign {
    my ($self, $value) = @_;

    if ($self->{st} =~ /^struct(\s+\w+)?$/) {
        my @v;
        if ($value->{t} eq "pointer") {
            push(@v, [ $value ]);
        } elsif ($value->{t} eq "struct") {
            $self->copy_struct($value);
            return $self;
        } else {
            push(@v, $value->{v});
        }
        $self->init_struct_from_ary(@v);
        return $self;
    }

    if (exists $value->{t}) {
        if ($self->{st} =~ /\[$/ && $value->{t} eq "pointer") {
            # 宣言で左辺が配列, 右辺が文字列リテラルのポインタ
            $self->assign_ary($value->deref->{buf}->{mem});
            return $self;
        } elsif ($value->{t} eq "pointer") {
            # ポインタの場合は VAR_ENTRY も含めて複製が必要。
            # コピー前後の pointer で buffer のみ共有する。
            $value = $value->clone;
        } elsif ($value->{t} eq "array") {
            $self->assign_ary($value->{v});
            return $self;
        }
    }
    $self->{buf}->{mem}->[ $self->{index} ] = $value;
    return $self;
}

sub assign_ary {
    my $self = shift;
    my $ary = shift;
    (my $type = $self->{st}) =~ s/\.\[//g;

    my @size;
    if ($self->{buf}->{size}) {
        @size = @{$self->{buf}->{size}};
    }

    my $var = $self->clone;
    if ($ary->[0]->{t} eq "array") {
        unless (defined $size[0]) { # 最初のサイズが不明な場合の計算
            $var->{buf}->{size}->[0] = $size[0] = int(@$ary);
        }
        &array_copy($var, $ary, $type, @size);
    } else {
        unless (defined $size[0]) { # 最初のサイズが不明な場合の計算
            my $ms = 1;
            map($ms *= $_, @size[1..$#size]);
            $size[0] = @$ary / $ms;
            $size[0] = int($size[0]) + ($size[0] > int($size[0]) ? 1 : 0);
            $var->{buf}->{size}->[0] = $size[0];
        }
        &array_flat_copy($var, $ary, $type, @size);
    }
}


sub array_flat_copy {
    my ($var, $ary, $type, @size_list) = @_;

    my $size = 1;
    map($size *= $_, @size_list);

    my $init_size = int(@$ary);
    my $min = ($size < $init_size ? $size : $init_size);

    my $i;
    for ($i = 0; $i < $min; $i++) {
        $var->assign( $ary->[$i] )->succ;
    }
    for ( ; $i < $size; $i++) {
        $var->assign( VALUE->literal(0, $type) )->succ;
    }
}


sub array_copy {
    my ($var, $ary, $type, @size_list) = @_;

    my $size = shift(@size_list);
    my $init_size = int(@$ary);
    my $min = ($size < $init_size ? $size : $init_size);

    my $i;
    for ($i = 0; $i < $min; $i++) {
        my $v = $ary->[$i];
        if ($v->{t} eq "array") {
            &array_copy($var, $v->{v}, $type, @size_list);
        } else {
            $var->assign($v)->succ;
        }
    }

    for (; $i < $size; $i++) {
        &array_zero($var, $type, @size_list);
    }
}

sub array_zero {
    my ($var, $type, @size_list) = @_;

    if (@size_list) {
        my $size = shift(@size_list);
        for (my $i = 0; $i < $size; $i++) {
            &array_zero($var, $type, @size_list);
        }
    } else {
        $var->assign( VALUE->literal(0, $type) )->succ;
    }
}

sub init_struct_from_ary {
    my $self = shift;
    my $ary = shift;

    $self->alloc_struct;
    my $struct = $self->value;

    foreach my $mn ( $struct->all_members_name ) {
        $struct->member($mn)->assign(shift @$ary);
    }
}

sub alloc_struct {
    my $self = shift;

    return $self if ($self->{buf}->{mem}->[ $self->{index} ]);

    my $sym_tbl = $self->{buf}->{vartbl}->{symtbl};

    my ($tag) = ($self->{st} =~ /^struct\s*(\w+)/);
    die "No tag exists in $self->{st}.\n" unless $tag;
    my $scope = $sym_tbl->{scope}->{ $self->{buf}->{scope} };
    die "No scope exists for $self->{buf}->{name}.\n" unless $scope;

    my $tag_info = $scope->lookup("tag:$tag");
    my @mem = @{$tag_info->{'#member'}};

    my $struct = VALUE::STRUCT->new(@mem);
    $self->{buf}->{mem}->[ $self->{index} ] = $struct;

    my $vartbl = $self->{buf}->{vartbl};

    # 構造体が入れ子になっているとき、代入される値は構造化されていることが前提。
    # フラットなデータの代入には対応できていない。
    state $id++; # 構造体ごとにメンバ名を区別するための id
    foreach my $m (@mem) {
        my $var = $vartbl->alloc_var_entry($self->{buf}->{name}."$id:$m",
                                           $tag_info->{$m},
                                           $self->{buf}->{scope});
        $struct->{v}->{$m} = $var;
    }
    return $self;
}

sub copy_struct {
    my $var = shift;
    my $src = shift;

    $var->alloc_struct unless defined $var->value_no_check;

    foreach my $m ($src->all_members_name) {
        my $r_mem = $src->member($m);
        my $l_mem = $var->value->member($m);
        if ($l_mem->{st} =~ /\[$/) {
            $l_mem->assign_ary($r_mem->{buf}->{mem});
        } else {
            $l_mem->assign($r_mem->value);
        }
    }
}


sub value_no_check {
    my ($self) = @_;
    my $res = $self->{buf}->{mem}->[ $self->{index} ];
    return $res;

}

sub value {
    my ($self) = @_;
    $Data::Dumper::Maxdepth = 3;
    if (exists $self->{buf}->{max_num} &&
            $self->{index} >= $self->{buf}->{max_num}) {
        die "**** Runtime Error: access to the outside of the buffer.\n**** "
            . $self->str . "\n\n";
    }
    my $res = $self->value_no_check;
    unless (defined $res) {
        die "**** Runtime Error: read the uninitilized variable.\n**** "
            . $self->str . "\n\n";
    }
    return $res;
}

sub succ {
    my $self = shift;
    $self->{index}++;
    return $self;
}

sub pred {
    my $self = shift;
    $self->{index}--;
    return $self;
}

sub clone {
    my $self = shift;
    return bless { %$self };
}

sub str {
    my $self = shift;
    my $name = $self->{buf}{name} // "#NONE";
    return join("", ("VAR_ENTRY $name at $self->{index} in [ ",
                     join(", ", map($_ ? $_->str : "undef", @{$self->{buf}->{mem}})),
                     " ] : $self->{st}"));
}

1;

##########################################################################
package VALUE;

use Data::Dumper;

sub new {
    return bless { };  # NIL
}

sub clone {
    my $self = shift;
    return bless { %$self };
}

sub is_nil {
    return !exists $_[0]->{t};
}

sub literal {
    my $self = shift;
    my ($val, $type) = @_;

    state %repo;
    my $sig =  "literal:$val:$type";
    return $repo{$sig} ||= bless { t => "literal", st => $type, v => $val };
}

sub literal_str {
    my ($self, $str, $vtbl) = @_;

    state %repo;

    my $var = ($repo{$str} ||= $vtbl->alloc_var_entry(undef, 'char.['));

    # eval でバックスラッシュ等の処理を行う
    $var->assign( $self->array_from_string( eval("$str") ));
    return VALUE::POINTER->new_from_ary($var);
}

sub op {
    my ($val1, $sym, $val2) = @_;

    if ($val2 && $val1->{t} eq "literal" && $val2->{t} eq "pointer") {
        # ポインタ演算。加算のときだけ、この計算が存在しうる。
        # 減算のときは正負が逆になるが、そもそも負のポインタはありえない。
        return $val2->op($sym, $val1);
    }

    $sym =~ s/_/$val1->{v}/;
    if ($val2) {
        $sym =~ s/_/$val2->{v}/;
    }
    my $res = eval($sym);
    $res = 0 if ($res eq "");
    return VALUE->literal($res, $val1->{st});
}

sub array {
    my $self = shift;
    my @vals = @_;

    return bless { t => "array", v => [ @vals ],  st => "array" };
}

sub array_from_string {
    my $self = shift;
    my $str = shift;
    return $self->array(
        map(VALUE->literal(ord($_), 'char'), split('', $str), "\0") );
}

sub mark {
    my $self = shift;
    return bless { t => $_[0] };
}

sub inc {
    my $self = shift;
    return VALUE->literal($self->{v} + 1, $self->{t});
}

sub dec {
    my $self = shift;
    return VALUE->literal($self->{v} - 1, $self->{t});
}

sub str {
    my $self = shift;
    my $raw_char = shift;
    return '#NIL' if $self->is_nil;
    if ($self->{t} eq "literal") {
        if ($self->{st} eq 'char' && !$raw_char) {
            my $v = $self->{v};
            $v = $v == 0 ? "\\0" : chr($v);
            return "'$v'";
        }
        return "". $self->{v};
    } elsif ($self->{t} eq "array") {
        return join('', "[ ", join(", ", map($_->str, @{$self->{v}})), " ]");
    } elsif ($self->{t} =~ /^#/) {
        return $self->{t};
    } else {
        $Data::Dumper::Maxdepth = 3;
        return Dumper($self);
    }
}

1;


###########################################################################
## ポインタとそれに関わる演算
package VALUE::POINTER;

use Data::Dumper;

sub new {
    my $self = shift;
    my $var = shift;

    return bless { t => "pointer", p => $var, st => $var->{st} . ".*",
                   size => [ 1 ] };

}

sub new_from_ary {
    my $self = shift;
    my $var = shift;

    ( my $t = $var->{st} ) =~ s/\[$/*/;

    my $size = [ @{$var->{buf}->{size}} ];
    shift(@$size);
    push(@$size, 1);
    return bless { t => "pointer", p => $var, st => $t, size => $size };
}

sub clone {
    my $self = shift;
    my $clone = bless { %$self };
    $clone->{size} = [ @{$self->{size}} ];
    $clone->{p} = $self->{p}->clone;
    return $clone;
}

sub deref { # dereference of pointer
    my $self = shift;
    return $self->{p};
}

sub inc {
    my $self = shift;
    my $res = $self->clone;
    my $size = 1;
    $size *= $_ foreach @{$res->{size} // []};
    $res->{p}->{index} += $size;
    return $res;
}

sub dec {
    my $self = shift;
    my $res = $self->clone;
    my $size = 1;
    $size *= $_ foreach @{$res->{size} // []};
    $res->{p}->{index} -= $size;
    return $res;
}

sub op {
    my ($ptr, $sym, $val2) = @_;
    # $ptr は pointer、$val2 は pointer or literal

    my $res;
    if ($sym eq "_+_" && $val2->{t} eq "literal") {
        $res = $ptr->clone;
        my $size = 1;
        if (exists $res->{size}) {  # 多重配列の場合
            $size *= $_ foreach @{$res->{size}};
        }
        $res->{p}->{index} += $val2->{v} * $size;

    } elsif ($sym eq "_-_") {
        if ($val2->{t} eq "pointer") {
            $res = VALUE->literal($ptr->{p}->{index} - $val2->{p}->{index},
                "int"); # size_t が正しい?
        } else { # i.e. literal
            $res = $ptr->clone;
            $res->{p}->{index} -= $val2->{v};
        }
    } else {
        $Data::Dumper::Maxdepth = 3;
        die "Illegal operation for pointer: $sym\n"
            . Dumper($ptr, $val2);
    }
    return $res;
}

sub str {
    my $self = shift;
    if (exists $self->{st} && $self->{st} eq "char.*") {
        my $var = $self->{p};
        my @s;
        for (my $i = $var->{index}; $var->{buf}->{mem}->[$i]; $i++) {
            last if ($var->{buf}->{mem}->[$i]->{v} == 0);
            push(@s, chr($var->{buf}->{mem}->[$i]->{v}));
        }
        return '"' . join('', @s) . '"';
    } else {
        my $name = $self->{p}{buf}{name} // "#NONE";
        my $index = $self->{p}{index};
        return "Pointer to ${name}[${index}] :" . $self->{p}->{st}
          . " size:[" . join(", ", @{$self->{size}}) . "]";
    }
}

1;

###########################################################################
package VALUE::STRUCT;
use Data::Dumper;

sub new {
    my $self = shift;
    return bless { t => "struct", v => undef, member => [ @_ ] };
}

sub member {
    my $self = shift;
    my $member = shift;
    return $self->{v}->{$member};
}

sub all_members_name {
    my $self = shift;
    return @{$self->{member}};
}

sub str {
    my $self = shift;
    return "{ ".
        join(", ", map("$_=".$self->{v}->{$_}->str, keys %{$self->{v}}))
        . " }";
}

1;

###########################################################################
package VALUE::FUNC;
use Data::Dumper;

sub new {
    my $self = shift;
    my $ptr = shift;
    return bless { t => "func", p => $ptr };
}

sub call {
    my $self = shift;
    my $visitor = shift;

    if (ref $self->{p} eq "CODE") {
        my @parts = split('_', $visitor->label);
        my @args = ();
        while (@parts > 2) {
            pop(@parts);
            unshift(@args, $visitor->{stack}->pop);
        }
        return $self->{p}->(@args);
    } else {
        my $call_node = $visitor->{cfg}->node_begin($self->{p});
        $visitor->call_to($call_node);
    }
}

sub str {
    my $self = shift;
    return "FuncCall( $self->{p} )";
}

1;

###########################################################################
package VALUE::MEMBER;
use Data::Dumper;

sub new {
    my $class = shift;
    my ($l, $r, $name) = @_;
    return bless { l => $l, r => $r, name => $name };
}

sub str {
    my $self = shift;
    return "MEMBER( $self->{name} )";
}

###########################################################################
# 記号表
package SYMTBL;

use Data::Dumper;

sub new {
    my $class = shift;
    my $ast = shift;

    my $obj = bless { "sym_tbl" => $ast->{sym}, scope => {} };
    &build_scope_map($obj->{sym_tbl}, $obj->{scope}, $obj);
    return $obj;
}

sub build_scope_map {
    my $sym_tbl = shift;
    my $map = shift;
    my $parent = shift;

    my $obj = bless { "sym_tbl" => $sym_tbl, "p" => $parent };
    $map->{ $sym_tbl->{'#id'} } = $obj;

    foreach (@{$sym_tbl->{scopes}}) {
        &build_scope_map($_, $map, $obj);
    }
}

sub lookup {
    my $self = shift;
    my $sym = shift;
    if (exists $self->{sym_tbl}->{$sym}) {
        return $self->{sym_tbl}->{$sym};
    }
    if ($self->{p}) {
        return $self->{p}->lookup($sym);
    }
    return undef;
}

1;

###########################################################################
# C の標準関数のシミュレーション
package C;
use Data::Dumper;

sub func_list {
    return [
        { name => "printf", type => "int", func => \&printf },
        { name => "scanf", type => "int", func => \&scanf },
        ];
}


sub printf {
    my $args = join(", ", map($_->str(1), @_));
    $args =~ s/\n/\\n/g;
    return eval("CORE::printf($args)");
}

sub scanf {
    my $fmt = shift;
    my @args = @_;

    my $fmt_str = eval($fmt->str);

    my @type = ($fmt_str =~ /%(\w)/g);

    my $fmt_re = $fmt_str;
    $fmt_re =~ s/%d/(\\d+)/g;
    $fmt_re =~ s/%s/(\\S+)/g;

    my @res = (<> =~ /^$fmt_re\s+/);

    foreach my $v (@args) {
        my $var = $v->deref;
        my $value = shift @res;
        my $type = shift @type;

        die "Illegal input for scanf.\n" unless defined $value;

        my $literal;
        if ($type eq "d") { # 整数リテラル
            $literal = VALUE->literal(int($value), "int");
        } elsif ($type eq "s") { # 文字列リテラル
            $literal = VALUE->array_from_string($value);
        }
        $var->assign($literal);
    }
    return 0;
}

1;
