#!/usr/bin/env perl
# 
# Copyright (c) 2009-2025 The TEBA Project. All rights reserved.
# 
# Redistribution and use in source, with or without modification, are
# permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# Author: Atsuhi Yoshida


use warnings;
use strict;

use FindBin qw($Bin);
use lib "$Bin/../TEBA";

# デバッグ用
use Data::Dumper;
use Carp qw(confess);
$SIG{__DIE__} = \&confess;

#use CParser;
use RewriteTokens2;
use CFG;
use CParserProc;

use Getopt::Std;

my %opts = ();
if (!getopts("hersdTC", \%opts) || $opts{h}) {
    print STDERR "c_peval.pl [-erdh] [TEBA Token file]\n",
	"  -e: replace expressions with evaluated values.\n",
	"  -r: remove redundant statements.\n",
	"  -s: start evaluation from the top.\n",
	"  -T: output the result as source text.\n",
	"  -C: parse input as C source code.\n",
	"  -d: debug mode.\n",
	"  -h: help.\n";
    exit(1);
}


# 構文解析した字句列を読み込み (JSON形式のASTではないよ!)
# オプション -C のときは、ソーステキストとして扱い、構文解析を適用。
my $tk_ast = $opts{C} ? [ CParserProc->new->parse(join('',<>)) ] : [ <> ];

# マークされた文に SP_TARGET を追加。マークは対象の文の直後につけた "/*E@*/" または "//E@"
my $rule = q( @ANY => { [ $:/./ ]* }
  # マークの前に存在する最初の B_ST または B_DE
  { $b:/B_(ST|DE)/ [any: [ $:/B_(ST|DE)/ ]! | $:/./ ]*  $:SP_C/..E@.*/ } => { $b '':SP_TARGET $any }
  # 構文解析の邪魔になるので、B_ST または B_DE の直前に移動
  { ($b:/B_(ST|DE)/ $s:SP_TARGET $st:@ANY $e:/E_(ST|DE)/) } => { $b $st $s $e }
  # 式文のセミコロンよりは前
  { $sc:SC $s:SP_TARGET } => { $s $sc }
  # この rule と find_target_node は整合していないといけない。
);

# 書換えルールを適用
my $rw = RewriteTokens->new()->rule($rule);
$rw->apply($tk_ast);

# AST の構築と CFG の生成
my $ast = AST::NODE->build_from_tokens($tk_ast);
my $cfg = CFG->new($ast);

# 解析開始位置の決定
my @target_node = (exists $opts{s} ? $cfg->node_begin : &find_target_node($cfg));
die "No target" unless @target_node;

# 各開始位置から解析を実行
&interpret($_) foreach (@target_node);

# オプションに応じてソースコードに反映
if ($opts{e}) {
    &replace_peval_value($ast);
} else {
    &apply_peval_value($ast);
}
if ($opts{r}) {
#    $cfg->add_parent_link_in_ast;
    &remove_redundant_statements($ast);
#    $cfg->remove_parent_link_ast;
}

# 字句列またはソーステキストに戻して出力
if ($opts{T}) {
    print $ast->str;
} else {
    print $ast->tokens;
}

# ToDo: ソースコードを出力する。

# ターゲットノードを探す
sub find_target_node {
    my $cfg = shift;
    my @res;
    # CFG のすべてのノードについて調べる
    foreach my $node ($cfg->node_all) {
	next if $node->label !~ /^#.*-begin$/;  # CFG の文などの begin のノードに限定
	my $ast_node = $node->ast_node;
	next if $ast_node->{t} !~ /^(ST_|DE)/;  # ターゲットは文という前提
	for (my $i = $#{$ast_node->{e}}; $i >= 0; $i--) {
	    if ($ast_node->{e}->[$i] =~ /^SP_TARGET/) {
		pop(@{$ast_node->{e}}); # remove SP_TARGET
		push(@res, $node);
	    }
	}
    }
    return @res;
}

# 解釈処理ごとに visitor を生成して実行
sub interpret {
    my $target_node = shift;
    # CFG 上を動く visitor を作成。解釈実行するので、順方向に移動する。
    my $visitor = $cfg->visitor->set_direction_next;

    # visitor に計算のための stack と、変数の表(var)を作成
    $visitor->{stack} = CFG::STACK->new; # visitor 間で共有する。
    $visitor->{var} = { 'printf:#GLOBAL' => PVAL->func('#printf') };
    $visitor->add_deep_copy_keys('var'); # 変数の表は visitor 間で共有しない。

    # 各ノードでの処理を行う手続きを登録
    $visitor->build(&peval_tbl);

    #$visitor->{env_sig} = sub { return "x"; }; # 1回のみ通過
    $visitor->set_loop_count(1); # 繰り返しは 1回のみ

    $visitor->{resumable} = \&resumable;
    $visitor->{resume} = \&resume;

    #$visitor->{env_sig} = undef;
    #$visitor->{clone_copy} = undef;
    # visitor は、繰返しでは同じノードを何度も通過するので、訪問記録は不要。
    # よって、env_sig は undef のままにする。
    # stack と var は、visitor 間で共有するので、shallow copy されればよい。
    # 複製のための特別な処理は不要なのえ、clone_copy も undef のままにする。

    # ターゲットノードに移動し、実行
    $visitor->move_to($target_node);
    $visitor->run;
}


# 部分実行器
sub peval_tbl {
    return [
	{   # 各ノードでの解析の先頭で行う共通処理 (主にデバッグ用)
	    begin => sub {
		return unless $opts{d};
		my $v = shift;
		print STDERR "----\nVISITOR: id = ", $v->id, "\n";
		print STDERR "Node: ", $v->label, "(", $v->label_type, ")\n";
		print STDERR "Stack ", Dumper($v->{stack});
		print STDERR "Var: ", Dumper($v->{var});
		print STDERR "\n";
	    }
	},
	{
	    p => "literal", # 定数値
	    a => sub {
		my $v = shift;
		my $value = $v->ast_node->attr('value');
		$value =~ s/\\\\/\\/g if $value =~/^"/; # 文字列の場合のエスケープ処理
		$v->{stack}->push(PVAL->num($value));
	    }
	},
	{
	    p => "rvalue", # 変数の値の参照(右辺値)
	    a => \&push_rvalue

	},
	{
	    p => "lvalue", # 左辺値 (変数名を使用)
	    a => \&push_lvalue
	},
	{
	    p => "lrvalue", # increment など、左辺値と右辺値の両方がある場合
	    a => sub {
		my $v = shift;
		&push_lvalue($v);
		&push_rvalue($v);
	    }
	},
	{
	    p => "assign: _=_", # 代入の場合
	    a => sub {
		my $v = shift;
		my $rvalue = $v->{stack}->pop;  # 右辺の値
		my $var = $v->{stack}->pop;     # 左辺の変数
		&assign_value($v, $var, $rvalue, $rvalue);  # 代入
	    }
	},
	{
	    p => "assign: _++", # 後置インクリメント
	    a => sub {
		my $v = shift;
		my $rvalue = $v->{stack}->pop;  # 右辺の値
		my $var = $v->{stack}->pop;     # 左辺の変数
		&assign_value($v, $var, $rvalue, $rvalue->add(1)); # 1を加算した値を代入
	    }
	},
	{
	    p => "assign: ++_", # 前置インクリメント
	    a => sub {
		my $v = shift;
		my $rvalue = $v->{stack}->pop;  # 右辺の値
		my $var = $v->{stack}->pop;     # 左辺の変数
		&assign_value($v, $var, $rvalue->add(1), $rvalue->add(1)); # 1を加算した値を代入
	    }
	},
	{
	    p => "op", # 代入以外の演算子
	    a => sub {
		my $v = shift;
		$v->ast_node->set('peval', PVAL->apply_op($v->ast_node, $v->{stack}));    # 計算して、式に評価値を記録
		$v->{stack}->push($v->ast_node->attr('peval'));                # スタックに評価値をプッシュ
	    }
	},
	{
	    p => "call", # 関数呼出し (どうしよう?)
	    a => sub {
		my $v = shift;
		die "Not supported yet: call.";
	    }
	},
	{
	    p => [ "#if-end", "#loop_end-in", # if 文を抜けるときと loop 終了時
		   "logical_op", "label" ], # 論理積, 論理和, ラベル文
	    a => sub {
		my $v = shift;
		$v->suspend;  # 他の経路が終わるまで待つ
	    }
	},
	{   # 繰り返しには対応できない
	    p => [ qr/^#(while|do)-begin/, qr/^#for_init-end/,  # ループの入口
		   qr/^#(loop-back|for_succ-end)/, # 条件に戻るとき
		   qr/^#loop_end-in/, # ループを抜けるとき
		],
	    a => sub {
		my $v = shift;
#		&save_var_table_in_ast($v);
		$v->{var} = {}; # 繰り返しの出入りで変数の表はリセット
	    }
	},
	{
	    p => qr/^#(expr_st|for_(init|succ))-end/,
	    a => sub {
		my $v = shift;
		# 式文、for 文の初期化式、前進式の計算結果がスタックに残っているので、削除 (まだ抜けがあると思われる)
		$v->{stack}->pop;
	    }
	}
	];
}

sub push_rvalue {
    my $v = shift;
    my $vname = $v->ast_node->uniq_varname;
    if (exists $v->{var}->{$vname}) {  # 変数が登録されていれば、その値を利用
	$v->ast_node->set('peval', $v->{var}->{$vname});
	$v->{stack}->push($v->{var}->{$vname});
    } else {
	$v->{stack}->push(PVAL->nonval());  # 不確定な値
    }
}

sub push_lvalue {
    my $v = shift;
    $v->{stack}->push(PVAL->lvar($v->ast_node));
}

sub assign_value {
    my ($v, $var, $rvalue, $assign_value) = @_;
    # 左辺値が記号表にあり、右辺が値なら代入を記録
    # そうでなければ、記号表から左辺値を削除
    if ($rvalue->is_num) {  # 右辺が計算された値であるなら
	$v->{var}->{$var->{n}->uniq_varname} = $assign_value;  # 記号表に登録
	#  $var->{n}->{peval} = $rvalue;      # 左辺の評価値を記録(しない)
	$v->{stack}->push($rvalue); # 左辺の変数の値が全体の評価値 (本当は down-cast される可能性がある)
    } else {                      # 右辺が計算された値でないなら、
	delete $v->{var}->{$var->{n}->uniq_varname}; # 記号表から左辺の変数を削除
	$v->{stack}->push(PVAL->nonval());  # 不確定な値
    }
    $v->ast_node->set('peval', $v->{stack}->top);  # 代入式の評価値を記録
}

# resume できる条件
sub resumable {
    my @v = @_;  # 待機中の visitors

    my $node = $v[0]->{cur};
#    print STDERR "Resumable: ", int(@{$node->{prev}}), "==", int(@v), "\n";
#    print STDERR "   label = ", $v[0]->label, "\n";

    if ($v[0]->label eq "#loop_end-in") {
	do { # まず #cond-end へ移動
	    $node = $node->prev_one; # この間、分岐はない。
	} until ($node->label eq "#cond-end");
	$node = $node->pair_begin_one; # #cond-end から #cond-begin へ
	# print STDERR "DEBUG: reached $node->{label}.\n";
    }

    # 経路の数と待機している visitor が数が一致していれば、すべて到達
    return int(@{$node->{prev}}) == int(@v);
}

# resume 処理: 共通する変数だけの表にし、visitor を1つにする。
sub resume {
    my $self = shift;
    foreach my $v (@_) {
	foreach my $k (keys %{$self->{var}}) {
	    if (exists $v->{var}->{$k}
		&& $self->{var}->{$k}->equal($v->{var}->{$k})) {
		next;
	    }
	    delete $self->{var}->{$k};
	}
	$v->move_to_final_state;  # 不要な visitor は終了
    }
}


# AST に部分実行の結果を反映
sub apply_peval_value {
    my $root = shift;

    if ($root->{t} eq "ID_VF" && exists $root->{peval}) { # 変数参照だけ置き換え
	my $v = $root->{peval}->{v};
	$root->append_element("PEVAL <(=$v)>");
    }
    map(&apply_peval_value($_), $root->children_all);
}

# AST に部分実行の結果に置き換え
sub replace_peval_value {
    my $root = shift;

    if ($root->{peval} && $root->{peval}->is_num # 計算済みの値を持つ要素について
	&& ($root->op_symbol || "") !~ m|^_[-+%*/]?=_$|) { # (代入を除く)
	my $v = $root->{peval}->{v};
	$root->set_element("PEVAL <($v)>");
    }
    map(&replace_peval_value($_), $root->children_all);
}

# 無駄な文をできるだけ削除する
sub remove_redundant_statements {
    my $root = shift;

    # # 解釈終了時点の変数の値を出力 (代入を残すなら、これは不要)
    # if (0 && exists $root->{pevaltbl}) {
    # 	foreach my $label (keys %{$root->{pevaltbl}}) {
    # 	    my @out;
    # 	    while (my ($var, $values) = each %{$root->{pevaltbl}->{$label}}) {
    # 		$var =~ s/:#\w+$//;
    # 		push(@out, map("$var = $_->{v}", grep(&isObj($_), @$values)));
    # 	    }
    # 	    my $out = "ST_EVAL\t</*E*/". join("; ", @out) . ";>";
    # 	    if ($label eq "#then-out") {
    # 		my $then = $root->child("then");
    # 		my $n = &push_tokens_with_comp_blk($then, $out);
    # 		$then->{else} += $n + 1 if exists $then->{else};
    # 	    } elsif ($label eq "#else-out") {
    # 		if (exists $root->{else}) {
    # 		    my $else = $root->child("else");
    # 		    &push_tokens_with_comp_blk($else, $out);
    # 		} else {
    # 		    $root->{else} = @{$root->{e}} + 2;
    # 		    push(@{$root->{e}}, "CT_EL\t<else>", "SP_B\t< >",
    # 			 { t => "ST_COMP", e => [ "C_L\t<{>", $out, "C_R\t<}>"], id => 999},
    # 			 "SP_NL\t<\\n>");
    # 		}
    # 	    } else {
    # 		unshift(@{$root->{e}}, $out);
    # 	    }
    # 	}
    # }

    if ($root->{t} eq "ST_EXPR" && exists $root->{expr}) { # 式文で、式が計算済みなものは削除
	my $e = $root->child("expr");
	if (exists $e->{peval}) {
	    if (exists $e->{peval}->{v}) {
		splice(@{$root->{e}}, $root->{expr}, 1);
		delete $root->{expr};
	    }
	}
    } elsif ($root->{t} eq "ST_IF") {  # if文
	my ($cond, $then, $else) = map($root->child($_), "cond", "then", "else");
	if (exists $cond->{peval}->{v}) {  # 条件が計算済み
	    if ($cond->{peval}->{v}) {     # 条件が成立していたら
		%$root = %$then;           # if 文の中身を then 節のものにする。
	    } elsif ($else) {              # 条件が成立しておらず、かつ、else 節が存在するなら
		%$root = %$else;           # if 文の中身を else 節に置き換える
	    } else {                       # 条件が成立せず、else 節も存在しないなら、空文に置き換える
		%$root = ( id => $root->{id}, t => 'ST_EXPR', e => [ "SC\t<;>" ] ); # empty statement
	    }
	}
    }

    map(&remove_redundant_statements($_), $root->children_all);
}

# sub push_tokens_with_comp_blk {
#     my $el = shift;
#     my $out = shift;
#     if ($el->{t} eq "ST_COMP") {
# 	my $tail = @{$el->{e}};
# 	push(@{$el->{e}}, $out, $tail);
# 	return 1;
#     } else {
# 	my $st = { %$el };
# 	$el->{t} = "ST_COMP";
# 	$el->{e} = [ "C_L\t<{>", $st, $out, "C_R\t<}>" ];
# 	$el->{id} = 999;
# 	return 1;
#     }
# }

# sub save_var_table_in_ast {
#     my $self = shift;
#     my $label = $self->{cur}->{label};
#     my $ast_node = $self->{cur}->{ast_node};
#     while (my ($k, $v) = each %{$self->{var}}) {
# 	push(@{$ast_node->{pevaltbl}->{$label}->{$k}}, $v);
#     }
# };


# sub isObj {
# #    return ref($_[0]) eq "HASH";
#     return ref($_[0]) ne "";
# }

#####################################################################################


############### 部分計算用のクラス
package PVAL;

use Data::Dumper;

####constructors

sub num {  # 計算された値を表現
    my $class = shift;
    my $v = shift; # 値
    return bless { t => 'num', v => $v };
}

sub nonval {  # 不確定な値を表現
    return bless { t => 'nonval' };
}

sub lvar {  # 左辺値を表現
    my $class = shift;
    my $var = shift; # 値
    return bless { t => 'lvar', n => $var };
}

sub func {
    my $class = shift;
    my $f = shift;
    return bless { t => 'func', f => $f };
}

#### test functions

sub is_num {  # 値が計算された値かを判定
    return $_[0]->{t} eq "num";
}

sub equal { # 等しさ
    my ($a, $b) = @_;
#    return 1 if "$a" eq "$b";
    return 0 unless ($a->{t} eq $b->{t});

    if ($a->{t} eq 'nonval') {
	return 0;
    } elsif ($a->{t} eq 'num') {
	return $a->{v} == $b->{v};
    } elsif ($a->{t} eq 'lvar') {
	return $a->{n}->uniq_varname eq $b->{n}->uniq_varname;
    } elsif ($a->{t} eq 'func') {
	return $a->{f} eq $b->{f};
    } else {
	return 0;
    }
}


# 演算子の適用

sub apply_op {  # 演算した結果の値
    my $class = shift;
    my ($node, $stack) = @_;
    if ($node->op_symbol =~ /^_([^_]+)_$/) {  # 2項演算子のみ
	my $op = $1;
	my @val = ($stack->pop, $stack->pop);
	# 両方とも計算可能な値である必要がある。そうでなければ、結果は不確定な値
	return PVAL->nonval unless ($val[1]->is_num && $val[0]->is_num);
	my $expr = $val[1]->{v} . $op . $val[0]->{v};
	$expr = "$expr ? 1 : 0" if $op =~ /^[<>]=?|[!=]=$/;  # 条件式の場合は直接 eval できない
	my $res = eval($expr); # eval を使って計算 (手抜き)
	return PVAL->num($res); # 計算結果を値として返す
    } elsif ($node->op_symbol =~ /^[_,]+$/) { # カンマ演算子
	my $op = $node->op_symbol;
	my $val = $stack->pop while ($op =~ s/_//);
	return $val;
    } else {
	die "Not supported yet ($node->{sym}).";
    }
}

# 数値の加算
sub add {
    my ($pv, $val) = @_;  # $val は数値 (ToDo: PVAL のオブジェクトに拡張)
    my $res = bless { %$pv };
    $res->{v} += $val;
    return $res;
}
