#!/usr/bin/env perl
# 
# Copyright (c) 2009-2025 The TEBA Project. All rights reserved.
# 
# Redistribution and use in source, with or without modification, are
# permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# Author: Atsuhi Yoshida


use warnings;
use strict;

use Data::Dumper;

use FindBin qw($Bin);
use lib "$Bin/../TEBA";

use TEBA2TEXT;
use CFG;

#########################################################################
# Program Slicing

use Getopt::Std;
my %opts = ();
if (!getopts("hit:", \%opts) || $opts{h}) {
    print STDERR "slicer.pl [-hi] [-t target_id ] [json-based-cfg-file]\n",
        "  -i: print IDs for slice criteria.\n",
        "  -t : make slice for the target.\n",
        "  -h: help.\n";
    exit(1);
}
# オプション -i : スライス基準(criteria)となる変数に id を付けてソースコードを表示
# オプション -t : 指定されたスライス基準に対してスライシングを実行して、結果を表示
# オプションが指定されていない場合は何も出力しない。

# AST を読み込む
my $ast = AST::NODE->build_from_json_text(join('', <>));

# 読み込んだ AST から expression level の CFG オブジェクトを生成する
my $cfg = CFG->new($ast);

# オプション -i のとき
if ($opts{i}) {
    # $cfg からSLICER オブジェクトを作り、スライス基準を埋め込んだコードに変換して表示
    print SLICER->new($cfg)->code_with_criterias;
    exit 0;
}

# オプション -t がないときは終了
exit 0 unless $opts{t};


# オプション -t で指定された id をスライス基準とする。
my $criteria = $cfg->node($opts{t});
# スライス基準の id がCFG に存在しない id だった場合はエラーにする。
die "Illegal node id: $opts{t}" unless defined $criteria;

my $slicer = SLICER->new($cfg, $criteria);
$slicer->run;
print $slicer->slice_code;

exit 0;


#########################################################################
# SLICER クラスの定義
# 注意:基本的な定義に基づいて実装しており、ポインタや配列、構造体については
# 特に考慮していない。

package SLICER;

use Data::Dumper;

# SLICER オブジェクトの生成
sub new {
    my $self = bless {};
    my $class = shift;         # これは無視される
    $self->{cfg} = shift;      # スライシングを適用する CFG オブジェクト
    $self->{criteria} = shift;   # スライス基準の ID

    $self->{decr} = {};        # スライスに含まれる変数のリスト (宣言の出力用)

    return $self;
}

# SLICER の実行
sub run {
    my $self = shift;

    # CFG 上を移動する visitor オブジェクトを生成
    # Backword Slicing をするために、移動方向は prev にする
    my $visitor = $self->{cfg}->visitor->set_direction_prev;

    # Slicing 用の visitor が持つ独自の属性値の設定
    $visitor->{vars} = {};              # visitor ごとに持つ
    $visitor->{decr} = $self->{decr};   # 全 visitor に共通
    # vars には、visitor が各ノードにおいて、スライス基準の計算にかかわる
    # 変数を記録する。移動の経路によって異なるので、visitor ごとに用意する。
    # visitor は分岐ごとに生成されるが、最終結果をまとめる必要があるので、
    # 共通の slice のハッシュオブジェクトに CFG の node id を書き込む。
    # 直感的には、各visitor が求めた slice の和集合になる。
    # 最後に計算に関わる変数の宣言も含めて出力したいので、decr の
    # ハッシュオブジェクトに変数名を記録する。こちらも、各visitor が
    # 求めた変数の和集合になる。

    # visitor が各ノードで行う解析処理の設定
    $visitor->{analyzer} = sub {
        my $v = shift;
        my $cur = $v->cur;      # visitor が訪問中のノード

        if ($cur->{label} =~/^assign/) {  # 代入式のとき
            my $a_op = $cur->ast_node;  # 対応する AST のノード(代入演算子)
            # 右辺と左辺の式を取り出す
            my ($lhs, $rhs) = $a_op->children("operand");
            if (&is_assign($lhs, $v)) { # 計算に必要な変数への代入であれば
                $cur->ast_node->set_mark;  # スライスに入れるASTノードをマーク

                # この代入文について vars の更新を行う
                unless ($rhs) {  # ++ と -- のときは左辺は右辺の役割がある
                    $rhs = $lhs; $lhs = undef; # 左辺を右辺として扱う
                }
                # 代入された変数は、計算に関与しなくなるので、vars から取り除く
                &remove_left_vars($lhs, $a_op, $v);
                # 右辺の変数は計算に関わるので、vars に加える
                &add_right_vars($lhs, $rhs, $v);
            }
        } elsif ($cur->{label} =~ /^#(true|false)$/) { # 条件分岐のノード
            # 条件式が制御フローに関わるので、スライスに追加する
            $cur->ast_node->set_mark;   # スライスに入れるASTノードをマーク
            # 条件式の計算に関わる式を vars に加える
            &add_condition($cur->ast_node, $v);
        }
    };

    # DEBUG 用 終了する visitor の情報を表示したいときに使用
    # $visitor->{final} = sub {
    #     my $v = shift;
    #     printf STDERR "DEBUG: final: %s\n", $v->id;
    # };

    # visitor が訪問済のノードに残す足跡(signature)
    # 同じ signature があると、visitor はそこで終了する。
    $visitor->{env_sig} = sub {
        my $self = shift;
        return &var_sig($self->{vars});
    };
    # ここでは変数の状態(vars)を文字列化して登録する。
    # 変数が同じ状態で訪問したのであれば、その先の計算結果に変化はない。
    # 異なるなら、スライスに含まれるノードが増える可能性があるので、
    # 継続して visitor を動作させる。
    # ここの定義を失敗すると、ループを回るたびに、visitor が生成され続け、
    # visitor の動作が停止しなくなる。

    # 分岐ごとの visitor の複製の方法 (次の (1) または (2))
    # (1) 各 visitor が個別に持つ属性値だけ、オブジェクトを共有しないよう複製を生成する。
    # $visitor->{clone_copy} = sub {
    # 	my ($self, $clone) = @_;
    # 	$clone->{vars} = { %{$self->{vars}}}; # deep copy する
    # 	# $clone->{decr} = $self->{decr};   # 自動的にコピーされるので書かなくてよい
    # };
    # (2) Deep copy するキーを指定する。(clone_copy を書かなくてよい)
    $visitor->set_deep_copy_keys('vars');

    # visitor をスライス基準に移動させる
    $visitor->move_to($self->{criteria});
    # スライス基準はスライスに含める必要があるので、slice に登録する
    $self->{criteria}->ast_node->set_mark;
    # スライス基準で参照する変数は、計算が必要な変数なので、vars に登録する。
    #    $visitor->{vars} = { $self->{criteria}->ast_node->uniq_varname => 1 };
    &record_var($visitor->{vars}, $self->{criteria}->ast_node);
    &record_var($visitor->{decr}, $self->{criteria}->ast_node);

    # visotir を CFG 上で動かす
    $visitor->run;

    return $self;
}


# 変数とハッシュの処理(記録, 取得, 削除)
# 異なるスコープに同名の変数がある可能性があるので、
# それらを区別するために、uniq_varname を用いる。
sub record_var {
    my $tbl = shift;
    my $var_node = shift;
    $tbl->{$var_node->uniq_varname} = $var_node;
}

sub get_var {
    my $tbl = shift;
    my $var_node = shift;
    return $tbl->{$var_node->uniq_varname};
}

sub delete_var {
    my $tbl = shift;
    my $var_node = shift;
    delete $tbl->{$var_node->uniq_varname};
}


# 必要な変数の代入かどうかを判定する
sub is_assign {
    my $lhs = shift;
    my $v = shift;

    # 左辺が識別子ではない場合、つまり、式になっている場合は
    # そこに出現する変数は右辺値なので、代入とは判定しない。
    return 0 if ($lhs->{t} ne "ID_VF");
    # 左辺の変数が計算に必要な変数に含まれているかを返す
    return &get_var($v->{vars}, $lhs);
}

# 左辺の変数を vars の登録から削除する
sub remove_left_vars {
    my $lhs = shift;
    my $a_op = shift;
    my $v = shift;

    # 左辺がない(++ や --のとき)は何もしない。
    return unless $lhs;
    # 単純な代入のときは左辺の変数を削除する。
    # += などの複合演算子の場合は、左辺も右辺値になるので、削除しない。
    if ($lhs->{t} eq "ID_VF" && $a_op->op_symbol eq "_=_") {
        &delete_var($v->{vars}, $lhs);
    }
}

# 右辺値を vars に登録する
sub add_right_vars {
    my ($lhs, $rhs) = (shift, shift);
    my $v = shift;

    my @rvalue;  # 右辺値の変数のリスト
    # 左辺に右辺値がある場合は @rvalue に登録
    if (defined $lhs && $lhs->{t} ne "ID_VF") {
        push(@rvalue, &right_vars($lhs));
    }

    # 右辺の変数を @rvalue に登録
    push(@rvalue, &right_vars($rhs));

    # @rvalue を vars と decr の両方に登録
    map(&record_var($v->{vars}, $_), @rvalue);
    map(&record_var($v->{decr}, $_), @rvalue);
}

# 条件式内の変数を vars に登録する
sub add_condition {
    my $cnd = shift;
    my $v = shift;

    # 条件式内の右辺値となる変数を取り出す
    my @rv = &right_vars($cnd);
    # vars と decr に登録する。
    map(&record_var($v->{vars}, $_), @rv);
    map(&record_var($v->{decr}, $_), @rv);
}

# 式の中の右辺値の変数を再帰的に辿って取り出す
sub right_vars
{
    my $ast_node = shift;
    # 式で代入の場合は、右辺だけを辿る
    if ($ast_node->{t} eq "P" && $ast_node->op_symbol =~ /^_=_$/) {
        my ($lhs, $rhs) = $ast_node->children("operand");
        return &right_vars($rhs);
    } elsif ($ast_node->{t} eq "ID_VF") {  # 変数であればそれを返す
        return ($ast_node);
    }
    # 子要素があれば、それを辿って返す
    return (map(&right_vars($_), $ast_node->children_all));
}

# vars の signature の生成
sub var_sig {
    my $vars = shift;
    # 変数名をソートとすることで正規化したものを signature とする。
    return join("#", sort(keys %$vars));
}

#---------------------------------------------------------------------#

# スライスに含まれる文を出力する
sub slice_code {
    my $self = shift;

    # AST の根から辿って、スライスを構成する字句を取り出す
    my @tokens = $self->slice_tokens($self->{cfg}->ast);

    # 字句列をテキストに変換する
    return TEBA2TEXT->new->set_teba(\@tokens)->text;
}

sub slice_tokens {
    my $self = shift;
    my $el = shift;     # 処理する部分木の根
    my $print = shift;  # 部分木の子孫を出力するかどうかの設定

    my @res;            # 出力を保存するリスト

    if ($el->{t} =~ /^ST_/) { # 文の場合
        # 文の内部で表示すべき式が存在したら、文自体を表示する
        my $is_print = 0;       # 標準は表示をしない
        foreach ($el->elem) { # 直接持つ子要素または字句について
            if ($el->is_node($_)) {   # 子要素の場合
                my @c = $self->slice_tokens($_); # 子要素について再帰的に処理
                $is_print = 1 if @c; # 子要素に出力するべきものがあれば、文も出力と設定
                push(@res, @c);  # 子要素の結果を出力に追加
            } else {
                push(@res, $_)   # 字句を出力に追加
            }
        }
        @res = () unless $is_print; # 子要素に出力すべきものがなければ、すべて消す
    } else {  # 文以外
        # スライスに含まれる変数を持つ宣言、または、スライスに登録されている要素なら
        if ($el->{t} eq "DE" && &is_var_decl($el, $self->{decr}) || $el->marked) {
            $print = 1;  # 子孫も含めて出力する
        }
        foreach ($el->elem) {  # 内部に含む子要素と字句をすべて出力に加える
            if ($el->is_node($_)) {
                push(@res, $self->slice_tokens($_, $print));
            } else {
                push(@res, $_) if $print;
            }
        }
    }
    return @res;
}

# vars 内の変数の宣言かどうかの判定
sub is_var_decl {
    my $el = shift;
    my $decr = shift;

    if ($el->{t} eq "ID_VF") {  # 変数なら decr に登録されているかどうか返す
        return &get_var($decr, $el);
    }
    foreach ($el->children_all) { # 子要素について再帰的に調査
        return 1 if &is_var_decl($_, $decr);  # 1つでも見つかれば真を返す
    }
    return 0;  # 最後まで見つからなかったので偽を返す
}

#--------------------------------------------------------------------#

# スライス基準になる変数に id を付けてソースコードを生成
sub code_with_criterias {
    my $self = shift;
    # CFG のすべての node について調査
    foreach my $node ($self->{cfg}->node_all) {
        if ($node->{label} =~ /^l?rvalue:/  # 左辺値で、
            && $node->ast_node->{t} eq "ID_VF") { # かつ変数なら
            $node->ast_node->append_element("TARGET <:[$node->{id}]>");

        }
    }

    # 木構造を字句列に変換したものをテキストに変換
    return $self->{cfg}->ast->str;
}


#################################################################################
# AST::NODE を拡張: ノードに印をつけるためのメソッドを追加

package AST::NODE;

sub set_mark {
    my $self = shift;
    $self->{slice_mark} = 1;
}

# スライスに入れるASTノードかを判定
sub marked {
    my $self = shift;
    return exists $self->{slice_mark};
}
